# coding=utf-8
import typing

from google.protobuf import descriptor_pb2
from google.protobuf import json_format
from google.protobuf import text_format
from google.protobuf import descriptor_pool
from google.protobuf import message_factory
from google.protobuf import message


class ProtoSupport:

    def __init__(self, pool: descriptor_pool.DescriptorPool):
        self._pool: descriptor_pool.DescriptorPool = pool
        self._message_factory: message_factory.MessageFactory = message_factory.MessageFactory(self._pool)
        self._available_names: typing.Set[str] = set()

    @classmethod
    def from_fds(cls, fds: descriptor_pb2.FileDescriptorSet) -> "ProtoSupport":
        pool = descriptor_pool.DescriptorPool()

        file_by_name = {file_proto.name: file_proto for file_proto in fds.file}

        def _add_file(file_proto):
            for dependency in file_proto.dependency:
                if dependency in file_by_name:
                    _add_file(file_by_name.pop(dependency))
            pool.Add(file_proto)

        while file_by_name:
            _add_file(file_by_name.popitem()[1])

        return cls(pool)

    def get_message_object_by_name(self, name: str) -> message.Message:
        message_descriptor = self._pool.FindMessageTypeByName(name)

        prototype = self._message_factory.GetPrototype(message_descriptor)
        extensions = self._pool.FindAllExtensions(message_descriptor)
        for ext in extensions:
            prototype.RegisterExtension(ext)
        return prototype()

    def convert_protobuf_bytes_to_json(self, name: str, message_bytes: bytes) -> dict:
        prototype = self.get_message_object_by_name(name)
        prototype.ParseFromString(message_bytes)
        return json_format.MessageToDict(prototype, descriptor_pool=self._pool)

    def convert_json_to_protobuf_bytes(self, name: str, js_blob: bytes) -> bytes:
        prototype = self.get_message_object_by_name(name)
        prototype = json_format.Parse(js_blob, prototype, descriptor_pool=self._pool)
        return prototype.SerializeToString()

    def convert_js_dict_to_protobuf_bytes(self, name: str, js_dict: dict) -> bytes:
        prototype = self.get_message_object_by_name(name)
        prototype = json_format.ParseDict(js_dict, prototype, descriptor_pool=self._pool)
        return prototype.SerializeToString()

    def convert_protobuf_bytes_to_text(self, name: str, message_bytes: bytes) -> str:
        prototype = self.get_message_object_by_name(name)
        prototype.ParseFromString(message_bytes)
        return text_format.MessageToString(prototype, indent=2, descriptor_pool=self._pool)
