# coding=utf-8
import getpass
import pathlib
from typing import Callable

import click
import sys

from tasklet.experimental.cli import consts


# noinspection PyUnreachableCode
if False:
    import sandbox.common.upload as common_upload


def upload_resource(
    token: str,
    description: str,
    attributes: [str],
    arch: str,
    resource_type: str,
    owner: str,
    file_meta: "common_upload.MDSHandle.FileMeta"

) -> int:
    import sandbox.scripts.tools.upload.uploader as sandbox_uploader
    # noinspection PyShadowingNames
    import sandbox.common.upload as common_upload

    attrs = {
        k.strip(): v.strip()
        for k, v in (
            attr.split("=", 1) for attr in attributes
        )
    }
    attrs.setdefault("ttl", "90")  # increased ttl to ensure availability of tasklet resources
    resource_meta = common_upload.MDSHandle.ResourceMeta(
        type=resource_type,
        arch=arch,
        owner=owner,
        description=description,
        attributes=attrs,
        release_to_yd=False
    )
    uploader = sandbox_uploader.SandboxUploader(
        handle_class=common_upload.MDSHandle, resource_meta=resource_meta, auth=token,
        sandbox_url=consts.SANDBOX_URL, sandbox_proxy_url=consts.SANDBOX_PROXY_URL, payload_meta=[file_meta]
    )
    resource_id = uploader.upload(output=sys.stderr)
    return resource_id


def upload_arguments(func: Callable):
    """
        Decorator, which provides following attributes and options:
        * description: str
        * attributes: [str]
        * arch: str (from ctm.OSFamily)
        * resource_type: str
        * owner: str
        * file_meta: common_upload.MDSHandle.FileMeta
    """

    # noinspection PyShadowingNames
    def tasklet_upload_argument_wrapper(func: Callable):
        import sandbox.common.types.misc as ctm

        func = click.option(
            "-d", "--description", metavar="<description>", default="", help="Resource description."
        )(func)
        func = click.option(
            "-A", "--attribute", "attributes", metavar="<key>=<value>", multiple=True,
            help="Additional resource attributes."
        )(func)
        func = click.option(
            "-a", "--arch", metavar="<arch>", type=click.Choice(list(ctm.OSFamily)),
            default=ctm.OSFamily.ANY, help="Resource architecture."
        )(func)
        func = click.option(
            "-t", "--type", "resource_type", metavar="<resource_type>",
            default=consts.DEFAULT_RESOURCE_TYPE, help=f"Resource type (default: {consts.DEFAULT_RESOURCE_TYPE})."
        )(func)
        func = click.option(
            "-O", "--owner", metavar="<owner>", default=getpass.getuser(),
            help="Task/resource owner (default: current user)."
        )(func)
        func = click.argument(
            "file_meta", metavar="<tasklet_executable>",
            type=click.Path(exists=True, readable=True, path_type=pathlib.Path),
            callback=get_file_meta
        )(func)
        return func

    return tasklet_upload_argument_wrapper(func)


def get_file_meta(
    ctx: click.Context, param: click.Parameter, value: pathlib.Path
) -> "common_upload.MDSHandle.FileMeta":
    _, _ = ctx, param
    import sandbox.common.upload as common_upload

    return common_upload.MDSHandle.FileMeta(
        value.open("rb"), value.stat().st_size, value.name
    )
