import dataclasses
import enum
import pathlib
import re
import typing
from typing import Optional

import click
from marshmallow import exceptions as mmexceptions
from marshmallow import validate as mmvalidate
import marshmallow_dataclass
import yaml

from tasklet.experimental.cli import arcadia_support
from tasklet.experimental.cli import interfaces
import tasklet.api.v2.data_model_pb2 as data_model
import tasklet.api.v2.runtime_environment_pb2 as runtime_environment
import tasklet.api.v2.tasklet_service_pb2 as tasklet_service

DATA_SIZE_REGEXP = r"([\d]+)\s*((?:K|M|G|T|P)?B)"
ACE_SUBJECT_REGEXP = re.compile(r"^(user|abc):([A-Za-z0-9_-]{1,40})$")


class StorageType(str, enum.Enum):
    HDD = "hdd"
    SSD = "ssd"
    RAM = "ram"

    def to_protobuf(self) -> data_model.EStorageClass:
        return {
            self.HDD: data_model.EStorageClass.E_STORAGE_CLASS_HDD,
            self.SSD: data_model.EStorageClass.E_STORAGE_CLASS_SSD,
            self.RAM: data_model.EStorageClass.E_STORAGE_CLASS_RAM,
        }[self]


def to_bytes(size: str) -> int:
    size_parts = re.findall(DATA_SIZE_REGEXP, size, re.IGNORECASE)
    base, unit = size_parts[0]
    shift = ["B", "KB", "MB", "GB", "TB", "PB", "EB", "ZB"].index(unit.upper())
    return int(base) << (10 * shift)


@dataclasses.dataclass
class TaskletDescriptorSpec:
    @dataclasses.dataclass
    class DescriptorMeta:
        name: str
        namespace: str
        owner: str
        catalog: str
        description: Optional[str]
        tracking_label: str = "latest"

    @dataclasses.dataclass
    class DescriptorSpec:
        @dataclasses.dataclass
        class Executor:
            type: str  # TODO: restrict choice
            java_main_class: Optional[str] = None

        @dataclasses.dataclass
        class Container:
            @dataclasses.dataclass
            class WorkdirSetup:
                type: StorageType = dataclasses.field(metadata={"by_value": True})
                space: str = dataclasses.field(
                    metadata={"validate": mmvalidate.Regexp(DATA_SIZE_REGEXP, re.IGNORECASE)}
                )

            cpu_limit: int
            ram_limit: str = dataclasses.field(
                metadata={"validate": mmvalidate.Regexp(DATA_SIZE_REGEXP, re.IGNORECASE)}
            )
            workdir: WorkdirSetup

        @dataclasses.dataclass
        class NaiveIOSchema:
            schema_id: str = ""
            input_message: str = ""
            output_message: str = ""

        @dataclasses.dataclass
        class Environment:
            arc_client: bool = False
            sandbox_resource_manager: bool = False

        executor: Executor
        container: Container
        naive_schema: NaiveIOSchema
        environment: Environment = Environment()

    meta: DescriptorMeta
    spec: DescriptorSpec

    # hidden fields
    t_yaml_path: typing.Optional[str]

    @classmethod
    def from_yaml(cls, path: pathlib.Path) -> "TaskletDescriptorSpec":
        if not path.exists():
            raise click.UsageError("Tasklet descriptor required, but not found")

        rv = cls.from_bytes(path.read_bytes())
        rv.t_yaml_path = str(path)
        return rv

    @classmethod
    def from_bytes(cls, body: bytes) -> "TaskletDescriptorSpec":
        data = yaml.safe_load(body)
        try:
            return marshmallow_dataclass.class_schema(cls)().load(data)
        except mmexceptions.ValidationError as err:
            click.echo(f"Validation error: {err}", err=True)
            exit(1)


class TaskletDescriptor(interfaces.ITaskletDescriptor):
    state: typing.Optional[TaskletDescriptorSpec] = None

    def reset(self):
        self.state = None

    def load_from_yaml(self, path: pathlib.Path):
        if self.state is not None:
            raise RuntimeError("Double initialization of tasklet descriptor")
        self.state = TaskletDescriptorSpec.from_yaml(path)

    def load_from_bytes(self, body: bytes):
        if self.state is not None:
            raise RuntimeError("Double initialization of tasklet descriptor")
        self.state = TaskletDescriptorSpec.from_bytes(body)

    def make_tasklet_message(self) -> data_model.Tasklet:
        if self.state is None:
            raise RuntimeError("State not initialized")

        spec = self.state
        rep_url = ""
        if arcadia_support.is_t_yaml_locate_in_arc(pathlib.Path(spec.t_yaml_path)):
            rep_url = arcadia_support.locate_arc_root_by_t_yaml(pathlib.Path(spec.t_yaml_path))
        t_yaml_relative_path = arcadia_support.relative_arc_path_by_t_yaml(pathlib.Path(spec.t_yaml_path))
        return data_model.Tasklet(
            meta=data_model.TaskletMeta(
                name=spec.meta.name,
                namespace=spec.meta.namespace,
                account_id=spec.meta.owner,
            ),
            spec=data_model.TaskletSpec(
                catalog=spec.meta.catalog,
                tracking_label=spec.meta.tracking_label,
                source_info=data_model.SourceInfo(
                    spec_path=str(t_yaml_relative_path),
                    repository_url=str(rep_url)
                )
            )
        )

    def make_create_tasklet_message(self) -> tasklet_service.CreateTaskletRequest:
        if self.state is None:
            raise RuntimeError("State not initialized")

        spec = self.state
        rep_url = ""
        if arcadia_support.is_t_yaml_locate_in_arc(pathlib.Path(spec.t_yaml_path)):
            rep_url = arcadia_support.locate_arc_root_by_t_yaml(pathlib.Path(spec.t_yaml_path))
        t_yaml_relative_path=arcadia_support.relative_arc_path_by_t_yaml(pathlib.Path(spec.t_yaml_path))
        return tasklet_service.CreateTaskletRequest(
            name=spec.meta.name,
            namespace=spec.meta.namespace,
            account_id=spec.meta.owner,
            catalog=spec.meta.catalog,
            tracking_label=spec.meta.tracking_label,
            source_info=data_model.SourceInfo(
                spec_path=str(t_yaml_relative_path),
                repository_url=str(rep_url)
            )
        )

    def make_update_tasklet_message(self, expected_revision: int) -> tasklet_service.UpdateTaskletRequest:
        if self.state is None:
            raise RuntimeError("State not initialized")

        spec = self.state
        rep_url = ""
        if arcadia_support.is_t_yaml_locate_in_arc(pathlib.Path(spec.t_yaml_path)):
            rep_url = arcadia_support.locate_arc_root_by_t_yaml(pathlib.Path(spec.t_yaml_path))
        t_yaml_relative_path = arcadia_support.relative_arc_path_by_t_yaml(pathlib.Path(spec.t_yaml_path))
        return tasklet_service.UpdateTaskletRequest(
            tasklet=spec.meta.name,
            namespace=spec.meta.namespace,
            expected_revision=expected_revision,
            catalog=spec.meta.catalog,
            tracking_label=spec.meta.tracking_label,
            source_info=data_model.SourceInfo(
                spec_path=str(t_yaml_relative_path),
                repository_url=str(rep_url)
            )
        )

    def _make_build_spec(self, resource_id: int, description: str) -> data_model.BuildSpec:
        if self.state is None:
            raise RuntimeError("State not initialized")
        spec: TaskletDescriptorSpec = self.state
        return data_model.BuildSpec(
            description=description,
            compute_resources=data_model.ComputeResources(
                vcpu_limit=spec.spec.container.cpu_limit,
                memory_limit=to_bytes(spec.spec.container.ram_limit),
            ),
            launch_spec=data_model.LaunchSpec(
                type=spec.spec.executor.type,
                jdk=data_model.LaunchSpec.JDKOptions(main_class=spec.spec.executor.java_main_class),
            ),
            workspace=data_model.BuildSpec.Workspace(
                storage_class=spec.spec.container.workdir.type.to_protobuf(),
                storage_size=to_bytes(spec.spec.container.workdir.space),
            ),
            environment=runtime_environment.Environment(
                porto=runtime_environment.PortoEnvironment(
                    enabled=False,
                ),
                docker=runtime_environment.DockerEnvironment(
                    enabled=False,
                ),
                java=runtime_environment.JavaEnvironment(
                    jdk11=runtime_environment.JdkReq(
                        enabled=False,
                    ),
                    jdk17=runtime_environment.JdkReq(
                        enabled=False,
                    ),
                ),
                arc_client=runtime_environment.ArcClient(
                    enabled=spec.spec.environment.arc_client
                ),
                sandbox_resource_manager=runtime_environment.SandboxResourceManager(
                    enabled=spec.spec.environment.sandbox_resource_manager
                )
            ),
            payload=data_model.BuildSpec.Payload(
                sandbox_resource_id=resource_id,
            ),
            schema=data_model.IOSchema(
                simple_proto=data_model.IOSimpleSchemaProto(
                    schema_hash=spec.spec.naive_schema.schema_id,
                    input_message=spec.spec.naive_schema.input_message,
                    output_message=spec.spec.naive_schema.output_message,
                )
            )
        )

    def make_build_message(self, resource_id: int, description: str) -> data_model.Build:
        if self.state is None:
            raise RuntimeError("State not initialized")

        spec = self.state
        return data_model.Build(
            meta=data_model.BuildMeta(
                namespace=spec.meta.namespace,
                tasklet=spec.meta.name,
            ),
            spec=self._make_build_spec(resource_id, description),
        )

    def make_create_build_request(self, resource_id: int, description: str) -> tasklet_service.CreateBuildRequest:
        if self.state is None:
            raise RuntimeError("State not initialized")

        bs = self._make_build_spec(resource_id, description)
        return tasklet_service.CreateBuildRequest(
            namespace=self.state.meta.namespace,
            tasklet=self.state.meta.name,
            description=bs.description,
            compute_resources=bs.compute_resources,
            launch_spec=bs.launch_spec,
            workspace=bs.workspace,
            environment=bs.environment,
            payload=bs.payload,
            schema=bs.schema,
        )
