# coding=utf-8
import os
from typing import Callable

import inject
import pytest
import yaml

import yatest.common as yatest

from tasklet.experimental.cli import consts
from tasklet.experimental.cli import context
from tasklet.experimental.cli import interfaces
from tasklet.experimental.cli import tasklet_descriptor
from tasklet.experimental.tests.common import models
from tasklet.experimental.tests.common import server_mock
from . import corpus


@pytest.fixture(scope="function")
def simple_test_tasklet(tasklet_server: server_mock.TaskletServer) -> models.Tasklet:
    namespace, tasklet, owner = "foo_namespace", "foo_tasklet", "abc:foo"
    ns = tasklet_server.create_ns(namespace, owner)
    tl = tasklet_server.create_tasklet(ns, tasklet, None, owner)
    return tl


@pytest.fixture(scope="function")
def get_simple_test_tasklet_descriptor(simple_test_tasklet: models.Tasklet) -> Callable[[str, dict, dict], str]:
    tasklet_descriptor_meta_delta = {
        "namespace": simple_test_tasklet.NS,
        "name": simple_test_tasklet.Name
    }

    def get_patched_descriptor_file(filename: str, meta_delta: dict, spec_delta: dict) -> str:
        tasklet_root = yatest.test_output_path(filename)
        os.mkdir(tasklet_root)
        t_yaml_path = os.path.join(tasklet_root, "t.yaml")
        with open(t_yaml_path, "w") as f:
            parsed = yaml.safe_load(corpus.correct_t_yaml)
            for key, value in meta_delta.items():
                parsed["meta"][key] = value
            parsed["meta"].update(tasklet_descriptor_meta_delta)
            for key, value in spec_delta.items():
                parsed["spec"][key].update(value)
            f.write(yaml.safe_dump(parsed))
        return t_yaml_path

    return get_patched_descriptor_file


@pytest.fixture(scope="function")
def tasklet_context(tasklet_server: server_mock.TaskletServer) -> context.TaskletContext:

    def setup_injector(binder: inject.Binder):
        def descriptor_constructor() -> tasklet_descriptor.TaskletDescriptor:
            return tasklet_descriptor.TaskletDescriptor()

        binder \
            .bind(interfaces.ITaskletContext, context.TaskletContext()) \
            .bind_to_constructor(interfaces.ITaskletDescriptor, descriptor_constructor)

    inject.clear_and_configure(setup_injector)
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    ctx.cluster = consts.ClusterConfig(consts.UNITTEST.name, endpoint=tasklet_server.grpc_address)
    ctx._auth = False
    return ctx


pytest_plugins = (
    "tasklet.experimental.tests.common",
)
