# coding=utf-8
import json
from typing import Callable

import pytest
from click import testing as click_testing

from tasklet.experimental.cli import cmd_build
from tasklet.experimental.cli import consts
from tasklet.experimental.cli import sandbox
from tasklet.experimental.tests.common import models
from tasklet.experimental.tests.common import server_mock


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_build_create(
    get_simple_test_tasklet_descriptor: Callable[[str, dict, dict], str],
    registered_schema: str,
    output_format: str,
):
    ram_limit = "100MB", "100.0Mb", str(100 * 2 ** 20)
    tasklet_spec_delta = dict(container=dict(ram_limit=ram_limit[0]))
    t_yaml_path = get_simple_test_tasklet_descriptor(f"build_create_{output_format}", {}, tasklet_spec_delta)
    resource_id = 12345
    runner = click_testing.CliRunner()
    result = runner.invoke(
        cmd_build.build_subcommand,
        ["create", "-o", output_format, "-f", t_yaml_path, "--schema-id", registered_schema, str(resource_id),
         "--description", "foo-bar-baz-description"],
    )
    assert str(resource_id) in result.output
    if output_format == consts.Output.TABLE:
        assert ram_limit[1] in result.output
    else:
        assert ram_limit[2] in result.output


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_build_upload(
    get_simple_test_tasklet_descriptor: Callable[[str, dict, dict], str],
    registered_schema: str, output_format: str, monkeypatch
):
    def mock_upload(*args, **kwargs):
        return resource_id

    monkeypatch.setattr(sandbox, "upload_resource", mock_upload)
    ram_limit = "100MB", "100.0Mb", str(100 * 2 ** 20)
    tasklet_spec_delta = dict(container=dict(ram_limit=ram_limit[0]))
    t_yaml_path = get_simple_test_tasklet_descriptor(f"build_upload_{output_format}", {}, tasklet_spec_delta)
    resource_id = 12345
    runner = click_testing.CliRunner()
    result = runner.invoke(
        cmd_build.build_subcommand, [
            "upload", "-o", output_format, "-f", t_yaml_path, "--schema-id", registered_schema, t_yaml_path,
            "--build-description", "my build description from unittests!"
        ],
    )
    assert str(resource_id) in result.output
    if output_format == consts.Output.TABLE:
        assert ram_limit[1] in result.output
    else:
        assert ram_limit[2] in result.output


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_build_list(
    tasklet_server: server_mock.TaskletServer,
    simple_test_tasklet: models.Tasklet,
    registered_schema: str,
    output_format: str
):
    build_resource_ids = 12345, 54321
    builds = []
    for resource_id in build_resource_ids:
        builds.append(tasklet_server.create_build(
            simple_test_tasklet, registered_schema,
            dict(payload=dict(sandbox_resource_id=resource_id))
        ))
    runner = click_testing.CliRunner()
    result = runner.invoke(
        cmd_build.build_subcommand,
        ["list", "-o", output_format, f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}"],
    )
    assert result.exit_code == 0
    for build in builds:
        assert build.ID in result.output
    for resource_id in build_resource_ids:
        assert str(resource_id) in result.output
    if output_format == consts.Output.JSON:
        data = json.loads(result.output)
        assert len(data["builds"]) == 2
    # Check pagination and limits
    result = runner.invoke(
        cmd_build.build_subcommand,
        ["list", "-o", output_format, f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}", "-l", "1"],
    )
    assert result.exit_code == 0
    if output_format == consts.Output.JSON:
        data = json.loads(result.output)
        assert len(data["builds"]) == 1
        assert data["token"] != "1"
        remain_builds = {build.ID for build in builds}.difference(data["builds"][0]["meta"]["id"])
        page_token = data["token"]
        result = runner.invoke(
            cmd_build.build_subcommand,
            [
                "list", "-o", output_format, f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}",
                "-l", "1", "-pt", str(page_token)
            ],
        )
        assert result.exit_code == 0
        data = json.loads(result.output)
        assert len(data["builds"]) == 1
        assert data["token"] == "1"
        assert data["builds"][0]["meta"]["id"] in remain_builds


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_build_get(
    tasklet_server: server_mock.TaskletServer,
    simple_test_tasklet: models.Tasklet,
    registered_schema: str,
    output_format: str
):
    build_resource_id = 12345
    bld = tasklet_server.create_build(
        simple_test_tasklet, registered_schema,
        dict(payload=dict(sandbox_resource_id=build_resource_id))
    )
    runner = click_testing.CliRunner()
    result = runner.invoke(
        cmd_build.build_subcommand,
        ["get", "-o", output_format, bld.ID],
    )
    assert result.exit_code == 0
    assert bld.ID in result.output
    assert str(build_resource_id) in result.output
