# coding=utf-8
import json

import pytest

from click.testing import CliRunner

from tasklet.experimental.cli import cmd_label
from tasklet.experimental.cli import consts
from tasklet.experimental.tests.common import models
from tasklet.experimental.tests.common import server_mock


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_label_create_without_build(
    simple_test_tasklet: models.Tasklet,
    output_format: str
):
    label_name = "foo_label"
    runner = CliRunner()
    result = runner.invoke(
        cmd_label.label_subcommand,
        ["create", "-o", output_format, f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}:{label_name}"],
    )
    assert result.exit_code == 0
    assert label_name in result.output


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_label_create_with_build(
    tasklet_server: server_mock.TaskletServer,
    simple_test_tasklet: models.Tasklet,
    registered_schema: str,
    output_format: str
):
    label_name = "foo_label"
    bld = tasklet_server.create_build(simple_test_tasklet, registered_schema, None)
    runner = CliRunner()
    result = runner.invoke(
        cmd_label.label_subcommand,
        [
            "create", "-o", output_format, "-b", bld.ID,
            f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}:{label_name}"
        ],
    )
    assert result.exit_code == 0
    assert label_name in result.output
    assert bld.ID in result.output


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_label_list(
    tasklet_server: server_mock.TaskletServer,
    simple_test_tasklet: models.Tasklet,
    registered_schema: str,
    output_format: str
):
    build_resource_id = 12345
    bld = tasklet_server.create_build(
        simple_test_tasklet, registered_schema,
        dict(payload=dict(sandbox_resource_id=build_resource_id))
    )
    labels = (("foo_label", bld), ("bar_label", None))
    for label_name, build in labels:
        tasklet_server.create_label(simple_test_tasklet, label_name, build, None)
    runner = CliRunner()
    result = runner.invoke(
        cmd_label.label_subcommand,
        ["list", "-o", output_format, f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}"],
    )
    assert result.exit_code == 0
    for label_name, build in labels:
        assert label_name in result.output
        if build:
            assert build.ID in result.output
            if output_format == consts.Output.TABLE:
                assert str(build_resource_id) in result.output
    if output_format == consts.Output.JSON:
        data = json.loads(result.output)
        assert len(data["labels"]) == 2


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_label_get(
    tasklet_server: server_mock.TaskletServer,
    simple_test_tasklet: models.Tasklet,
    registered_schema: str,
    output_format: str
):
    label_name = "foo_label"
    build_resource_id = 12345
    bld = tasklet_server.create_build(
        simple_test_tasklet, registered_schema,
        dict(payload=dict(sandbox_resource_id=build_resource_id))
    )
    tasklet_server.create_label(simple_test_tasklet, label_name, bld, None)
    runner = CliRunner()
    result = runner.invoke(
        cmd_label.label_subcommand,
        ["get", "-o", output_format, f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}:{label_name}"],
    )
    assert result.exit_code == 0
    assert label_name in result.output
    assert bld.ID in result.output


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_label_move(
    tasklet_server: server_mock.TaskletServer,
    simple_test_tasklet: models.Tasklet,
    registered_schema: str,
    output_format: str
):
    label_name = "foo_label"
    build_resource_id_src, build_resource_id_dst = 12345, 54321
    bld_src = tasklet_server.create_build(
        simple_test_tasklet, registered_schema,
        dict(payload=dict(sandbox_resource_id=build_resource_id_src))
    )
    bld_dst = tasklet_server.create_build(
        simple_test_tasklet, registered_schema,
        dict(payload=dict(sandbox_resource_id=build_resource_id_dst))
    )
    tasklet_server.create_label(simple_test_tasklet, label_name, None, None)
    runner = CliRunner()
    result_initial = runner.invoke(
        cmd_label.label_subcommand, [
            "move", "-o", output_format, "--to", bld_src.ID,
            f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}:{label_name}",
        ],
    )
    assert result_initial.exit_code == 0
    assert label_name in result_initial.output
    assert bld_src.ID in result_initial.output
    result_follow = runner.invoke(
        cmd_label.label_subcommand, [
            "move", "-o", output_format, "--to", bld_dst.ID,
            f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}:{label_name}"
        ],
    )
    assert result_follow.exit_code == 0
    assert label_name in result_follow.output
    assert bld_dst.ID in result_follow.output
    result_retry = runner.invoke(
        cmd_label.label_subcommand, [
            "move", "-o", output_format, "--from", bld_src.ID, "--to", bld_dst.ID,
            f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}:{label_name}",
        ],
    )
    assert result_retry.exit_code != 0
