# coding=utf-8

import os

from click.testing import CliRunner
from google.protobuf import json_format
import pytest
from yatest import common as yatest

from tasklet.experimental.cli import consts
from tasklet.experimental.cli import main
from tasklet.api.v2 import well_known_structures_pb2
from tasklet.experimental.tests.common import server_mock


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_run(tasklet_server: server_mock.TaskletServer, registered_schema: str, output_format: str):
    namespace = tasklet_server.create_ns("default")
    tasklet = tasklet_server.create_tasklet(namespace, "TLFoo", {"catalog": "/home/alximik"})
    build = tasklet_server.create_build(tasklet, registered_schema, None)
    label = tasklet_server.create_label(tasklet, "latest", build, {})

    sample_input = well_known_structures_pb2.GenericBinary(
        payload=b"foo-bar-baz",
    )
    run_root = yatest.test_output_path(f"run_root_{output_format}")
    os.mkdir(run_root)

    runner = CliRunner()
    for payload_format in [consts.ProtoFormat.PROTO_BINARY, consts.ProtoFormat.PROTO_JSON]:
        input_file = os.path.join(run_root, f"input.{payload_format}")
        with open(input_file, "wb") as f:
            if payload_format == consts.ProtoFormat.PROTO_BINARY:
                f.write(sample_input.SerializeToString())
            elif payload_format == consts.ProtoFormat.PROTO_JSON:
                f.write(json_format.MessageToJson(sample_input).encode())
            else:
                raise RuntimeError(f"Unsupported input format: {payload_format}")

        result = runner.invoke(
            main.cli,
            ["run", "-o", output_format, "-i", payload_format, f"{tasklet.NS}/{tasklet.Name}:{label.Name}", input_file],
        )
        assert result.exit_code == 0
        if output_format == consts.Output.TABLE:
            assert "Task Status" in result.output
        else:
            assert "E_EXECUTION_STATUS_EXECUTING" in result.output
