# coding=utf-8
import json
import uuid
from typing import Callable

import inject
import pytest
from click.testing import CliRunner

from tasklet.experimental.cli import cmd_tasklet
from tasklet.experimental.cli import consts
from tasklet.experimental.cli import interfaces
from tasklet.experimental.cli import tasklet_descriptor
from tasklet.experimental.tests.common import models
from tasklet.experimental.tests.common import server_mock


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_tasklet_create(
    simple_test_tasklet: models.Tasklet,
    get_simple_test_tasklet_descriptor: Callable[[str, dict, dict], str],
    output_format: str
):
    t_yaml_path = get_simple_test_tasklet_descriptor(f"tasklet_create_{output_format}", {}, {})
    runner = CliRunner()
    result = runner.invoke(
        cmd_tasklet.tasklet_subcommand,
        ["create", "-o", output_format, "-f", t_yaml_path],
    )
    assert result.exit_code == 0
    assert simple_test_tasklet.Name in result.output


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_tasklet_list(tasklet_server: server_mock.TaskletServer, output_format: str):
    tasklets = ("foo_tasklet", "bar_tasklet")
    namespace_name, owner = "foo_namespace", "abc:foo"
    ns = tasklet_server.create_ns(namespace_name, owner)
    for tasklet_name in tasklets:
        tasklet_server.create_tasklet(ns, tasklet_name, None, owner)
    runner = CliRunner()
    result = runner.invoke(
        cmd_tasklet.tasklet_subcommand,
        ["list", "-o", output_format, namespace_name],
    )
    assert result.exit_code == 0
    for tasklet in tasklets:
        assert tasklet in result.output
    assert owner in result.output
    if output_format == consts.Output.JSON:
        data = json.loads(result.output)
        assert len(data["tasklets"]) == 2


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_tasklet_get(
    tasklet_server: server_mock.TaskletServer,
    simple_test_tasklet: models.Tasklet,
    output_format: str
):
    label_name = "foo_label"
    tasklet_server.create_label(simple_test_tasklet, label_name, None, None)
    runner = CliRunner()
    # NB: tasklet get also dumps list of labels in tasklet
    result = runner.invoke(
        cmd_tasklet.tasklet_subcommand,
        ["get", "-o", output_format, f"{simple_test_tasklet.NS}/{simple_test_tasklet.Name}"],
    )
    assert result.exit_code == 0
    assert simple_test_tasklet.Name in result.output
    if output_format == consts.Output.TABLE:
        assert label_name in result.output


@pytest.mark.usefixtures("tasklet_context")
@pytest.mark.parametrize("output_format", consts.ALL_OUTPUT_FORMATS)
def test_tasklet_update(
    simple_test_tasklet: models.Tasklet,
    get_simple_test_tasklet_descriptor: Callable[[str, dict, dict], str],
    output_format: str
):
    t_yaml_path = get_simple_test_tasklet_descriptor(f"tasklet_create_{output_format}_v1", {}, {})
    runner = CliRunner()
    result = runner.invoke(
        cmd_tasklet.tasklet_subcommand,
        ["create", "-o", output_format, "-f", t_yaml_path],
    )
    if result.exception:
        raise result.exception
    assert result.exit_code == 0
    assert simple_test_tasklet.Name in result.output

    new_catalog = "/".join(["", "home", "unitest", str(uuid.uuid4())])
    # NB: override t.yaml with same data with patched catalog
    new_t_yaml_path = get_simple_test_tasklet_descriptor(
        f"tasklet_create_{output_format}_v2",
        {"catalog": new_catalog},
        {},
    )
    # noinspection PyTypeChecker
    config: tasklet_descriptor.TaskletDescriptor = inject.instance(interfaces.ITaskletDescriptor)
    config.reset()  # NB: t.yaml changed

    result = runner.invoke(
        cmd_tasklet.tasklet_subcommand,
        ["update", "-o", output_format, "-f", new_t_yaml_path],
    )
    if result.exception:
        raise result.exception
    assert result.exit_code == 0
    assert simple_test_tasklet.Name in result.output
    assert new_catalog in result.output
