package configs

import (
	"embed"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/library/go/valid"
	"a.yandex-team.ru/tasklet/experimental/cmd/server/configs"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const (
	IdmConfigPrefix = "idm-"
	IdmConfigSuffix = ".yaml"
)

//go:embed *.yaml
var embeddedFS embed.FS

type IDMConfig struct {
	TVM tvm.ClientID `yaml:"tvm"`
	URL string       `yaml:"url" valid:"required"`
}

type CommonConfig struct {
	System string     `yaml:"system" valid:"required"`
	IDM    *IDMConfig `yaml:"idm"`
}

type IdmTreeConfig struct {
	Enabled    bool `yaml:"enabled"`
	SyncPeriod int  `yaml:"sync_period"`
}

type APIConfig struct {
	Port int `yaml:"port"`
}

type IdmServiceConfig struct {
	Common  *CommonConfig  `yaml:"common"`
	IdmTree *IdmTreeConfig `yaml:"idm_tree"`
	API     *APIConfig     `yaml:"api"`
}

func NewIdmConfig(configName string) (*IdmServiceConfig, error) {
	fullName := IdmConfigPrefix + configName + IdmConfigSuffix
	data, err := embeddedFS.ReadFile(fullName)
	if err != nil {
		return nil, xerrors.Errorf("Error on loading idm config. Err: %w", err)
	}
	idmConfig := &IdmServiceConfig{}
	unmarshalErr := yaml.Unmarshal(data, idmConfig)
	if unmarshalErr != nil {
		return nil, xerrors.Errorf("Error on unmarshal idm config. Err: %w", err)
	}
	return idmConfig, nil
}

func (conf *IdmServiceConfig) Validate() error {
	var validationCtx = valid.NewValidationCtx()
	validationCtx.Add(
		"required", valid.WrapValidator(
			func(s string) error {
				if len(s) == 0 {
					return xerrors.New("Value is required")
				}
				return nil
			},
		),
	)

	if err := valid.Struct(validationCtx, conf); err != nil {
		return xerrors.Errorf("Config validation failed: %+v", err)
	}
	return nil
}

type Config struct {
	TaskletServerConfig *configs.Config
	IdmConfig           *IdmServiceConfig
}

func (conf *Config) Validate() error {
	idmConfErr := conf.IdmConfig.Validate()
	if idmConfErr != nil {
		return idmConfErr
	}
	tsConfErr := conf.TaskletServerConfig.Validate()
	if tsConfErr != nil {
		return tsConfErr
	}
	return nil
}

func NewConfig(baseConfigName string, patchFilePath string) (conf *Config, err error) {
	taskletServerConfig, tscErr := configs.New(baseConfigName, patchFilePath)
	if tscErr != nil {
		return nil, tscErr
	}
	idmConfig, icErr := NewIdmConfig(baseConfigName)
	if icErr != nil {
		return nil, icErr
	}
	conf = &Config{TaskletServerConfig: taskletServerConfig, IdmConfig: idmConfig}
	return conf, nil
}
