package main

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/app"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics/collect"
	"a.yandex-team.ru/library/go/core/metrics/collect/policy/inflight"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/maxprocs"
	idmConfigs "a.yandex-team.ru/tasklet/experimental/cmd/idm/configs"
	serverConfigs "a.yandex-team.ru/tasklet/experimental/cmd/server/configs"
	"a.yandex-team.ru/tasklet/experimental/internal/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/locks"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/ydbstore"
	"a.yandex-team.ru/tasklet/experimental/internal/utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/idmclient"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/idm/services"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/tasklet/experimental/internal/ydbmigrate"
)

var (
	configFilePath string
	baseConfigName string
	conf           *idmConfigs.Config
)

func initConfig() (err error) {
	if conf, err = idmConfigs.NewConfig(baseConfigName, configFilePath); err != nil {
		return err
	}
	return nil
}

func validateConfig(_ *cobra.Command, _ []string) error {
	return conf.Validate()
}

func rootCmdHandler(_ *cobra.Command, _ []string) error {
	if err := initConfig(); err != nil {
		return err
	}
	maxprocs.AdjustAuto()
	return nil
}

func IdmCmdHandler(_ *cobra.Command, _ []string) error {
	loggers, err := lib.NewLoggers(conf.TaskletServerConfig.Logging)
	if err != nil {
		return err
	}

	rootCtx, rootCancel := context.WithCancel(context.Background())
	defer rootCancel()
	coreLog := loggers[lib.CoreLogger]

	solomonRegistry := solomon.NewRegistry(
		solomon.NewRegistryOpts().
			SetRated(true).
			AddCollectors(
				rootCtx,
				inflight.NewCollectorPolicy(),
				collect.GoMetrics,
				collect.ProcessMetrics,
			),
	)

	ydbClient, err := xydb.NewClient(
		rootCtx,
		conf.TaskletServerConfig.DB.Ydb,
		utils.MustToken(utils.LoadToken(conf.TaskletServerConfig.DB.Ydb.TokenPath)),
		coreLog.WithName("ydb_client"),
		solomonRegistry,
	)
	if err != nil {
		coreLog.Error("Failed to construct YDB client", log.Error(err))
		return err
	}

	defer func() {
		err := ydbClient.Close(rootCtx)
		if err != nil {
			coreLog.Warn("ydb close error", log.Error(err))
		}
	}()

	mc := ydbmigrate.NewMigrationClient(ydbClient, coreLog)
	migrateErr := mc.MigrationCheckLoop(rootCtx, ydbstore.SchemaVersion)

	if migrateErr != nil {
		coreLog.Error("Error on YDB migration check", log.Error(migrateErr))
		return migrateErr
	}

	var db storage.IStorage = ydbstore.NewStorage(ydbClient, coreLog.WithName("storage"))
	distributedLocksRepo := locks.NewLocksRepo(ydbClient)

	// Starting components
	errorChan := make(chan error, 4)
	stop := make(chan os.Signal, 1)
	signal.Notify(stop, syscall.SIGINT, syscall.SIGTERM)
	treePusherLogger := coreLog.WithName("treePusher")

	tvmClient, tvmErr := tvmtool.NewDeployClient(
		tvmtool.WithCacheEnabled(true),
		tvmtool.WithLogger(treePusherLogger.WithName("tvmtool").Structured()),
	)
	if tvmErr != nil {
		coreLog.Error("Unable to create TVM Client", log.Error(tvmErr))
		return err
	}

	idmClient := &idmclient.Client{
		Logger:      treePusherLogger,
		TVMClient:   tvmClient,
		TVMClientID: conf.IdmConfig.Common.IDM.TVM,
		URL:         conf.IdmConfig.Common.IDM.URL,
		Limit:       100,
		Silent:      false,
		NoMeta:      false,
		MasterOnly:  false,
		HTTPClient:  &http.Client{},
	}

	staffClient := &staff.Client{
		Logger:      coreLog.WithName("staff_client"),
		TVMClient:   tvmClient,
		TVMClientID: conf.TaskletServerConfig.Staff.TVM,
		URL:         conf.TaskletServerConfig.Staff.URL,
		HTTPClient:  &http.Client{},
	}
	staffCache := &staff.StaffGroupsCache{StaffClient: staffClient}

	treePusher, err := services.NewTreePusher(
		conf.IdmConfig,
		treePusherLogger,
		idmClient,
		&services.IdmTreeBuilder{Logger: treePusherLogger.WithName("treeBuilder"), DB: db},
		staffCache,
		locks.NewLocker(
			distributedLocksRepo,
			"idm_tree_pusher",
			conf.TaskletServerConfig.Installation,
			coreLog.WithName("idm_tree_pusher_locker"),
		),
	)

	if err != nil {
		coreLog.Error("Unable to load IDM tree pusher", log.Error(err))
		return err
	}

	go func() {
		// NB: let grpc server initialize
		<-time.After(1 * time.Second)
		coreLog.Infof("Starting IDM tree pusher")
		err := treePusher.Run()
		coreLog.Infof("IDM tree pusher stopped: %v", err)
		errorChan <- err
	}()

	serverLogger := coreLog.WithName("IDMServer")

	idmServer := app.NewIdmServer(
		conf, serverLogger, db, solomonRegistry, staffClient, staffCache, &app.Authorization{
			IdmTVM:    conf.IdmConfig.Common.IDM.TVM,
			TvmClient: tvmClient,
		},
	)
	go func() {
		// NB: let grpc server initialize
		coreLog.Infof("Starting IDM Server")
		err := idmServer.Serve()
		if err != nil {
			coreLog.Warnf("Server stopped. Err: %v", err)
		}
	}()

	select {
	case <-stop:
		ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
		defer cancel()
		if serverErr := idmServer.Stop(ctx); serverErr != nil {
			return serverErr
		}
		treePusher.Stop()
	case err := <-errorChan:
		// NB: emergency stop
		return err
	}
	return nil
}

var rootCmd = &cobra.Command{
	Use:               "tasklet-idm",
	Short:             "Tasklet Idm",
	PersistentPreRunE: rootCmdHandler,
	PreRunE:           validateConfig,
	RunE:              IdmCmdHandler,
}

func init() {
	rootCmd.PersistentFlags().StringVarP(
		&configFilePath,
		"config-path",
		"c",
		"",
		"path to config; it is used as patch for base config",
	)
	rootCmd.PersistentFlags().StringVarP(
		&baseConfigName,
		"base-config",
		"b",
		serverConfigs.DefaultEmbeddedConfig,
		fmt.Sprintf("name of base embedded config; values: %v", serverConfigs.ListEmbedded()),
	)
}

func main() {
	err := rootCmd.Execute()
	if err != nil {
		panic(err)
	}
}
