package configs

import (
	"embed"
	"os"
	"strings"

	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/valid"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/apiserver"
	"a.yandex-team.ru/tasklet/experimental/internal/handler"
	"a.yandex-team.ru/tasklet/experimental/internal/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/processor"
	"a.yandex-team.ru/tasklet/experimental/internal/revproxy"
	"a.yandex-team.ru/tasklet/experimental/internal/sbresolver"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/ytdriver"
)

const (
	DefaultEmbeddedConfig = "base"
	embeddedConfigsSuffix = ".yaml"
)

//go:embed *.yaml
var embeddedFS embed.FS

type InheritanceConfig struct {
	Parent string `yaml:"parent"`
}

type Config struct {
	Installation     string                    `yaml:"installation"`
	Proxy            *revproxy.ProxyConfig     `yaml:"proxy"`
	APIServer        *apiserver.Config         `yaml:"apiserver"`
	Handler          *handler.Config           `yaml:"handler"`
	Logging          *lib.LoggingConfig        `yaml:"loggers"`
	DB               *storage.Config           `yaml:"db"`
	Processor        *processor.Config         `yaml:"processor"`
	Sandbox          *sandbox.Config           `yaml:"sandbox"`
	GroupCacheConfig *sandbox.GroupCacheConfig `yaml:"sandbox_group_cache"`
	YTConfig         *ytdriver.Config          `yaml:"yt_config"`
	Executor         *apiclient.Config         `yaml:"executor"`
	ResourceResolver *sbresolver.Config        `yaml:"resource_resolver"`
	Staff            *staff.Config             `yaml:"staff"`
}

// New creates Config with given base config name and patch file
func New(baseConfigName string, patchFilePath string) (conf *Config, err error) {
	conf = &Config{}
	if err = conf.patchWithRecurrentConfig(baseConfigName); err != nil {
		return nil, err
	}

	if patchFilePath != "" {
		if err = conf.patchWithFile(patchFilePath); err != nil {
			return nil, err
		}
	}

	conf.applyVariableTransfers()
	return conf, err
}

func (conf *Config) applyVariableTransfers() {
	if conf.Proxy.Backend.Port == 0 {
		conf.Proxy.Backend.Port = conf.APIServer.Port
	}
}

// Dump dumps state of config to prettified yaml string
func (conf *Config) Dump() string {
	dump, _ := yaml.Marshal(conf)
	return string(dump)
}

func (conf *Config) patchWithFile(filePath string) error {
	if data, err := os.ReadFile(filePath); err != nil {
		return err
	} else {
		return yaml.Unmarshal(data, conf)
	}
}

func (conf *Config) patchWithRecurrentConfig(confName string) error {
	fullName := confName + embeddedConfigsSuffix

	if data, err := embeddedFS.ReadFile(fullName); err == nil {
		inheritance := &InheritanceConfig{}
		if err = yaml.Unmarshal(data, inheritance); err != nil {
			return err
		} else {
			if inheritance.Parent != "" {
				if err = conf.patchWithRecurrentConfig(inheritance.Parent); err != nil {
					return err
				}
			}
		}

		return yaml.Unmarshal(data, conf)
	} else {
		return err
	}
}

func (conf *Config) Validate() error {
	var validationCtx = valid.NewValidationCtx()
	validationCtx.Add("greater", valid.Greater)
	validationCtx.Add(
		"required", valid.WrapValidator(
			func(s string) error {
				if len(s) == 0 {
					return xerrors.New("Value is required")
				}
				return nil
			},
		),
	)

	if err := valid.Struct(validationCtx, conf); err != nil {
		return xerrors.Errorf("Config validation failed: %+v", err)
	}
	return nil
}

// ListEmbedded returns list of available embedded configs
func ListEmbedded() []string {
	var embedded []string
	items, _ := embeddedFS.ReadDir(".")
	for _, item := range items {
		if !item.IsDir() && strings.HasSuffix(item.Name(), embeddedConfigsSuffix) {
			name := item.Name()[0 : len(item.Name())-len(embeddedConfigsSuffix)]
			embedded = append(embedded, name)
		}
	}
	return embedded
}
