#!/usr/bin/env python3
import os
import shutil

import grpc

from sandbox.tasklet.sidecars.resource_manager.proto import resource_pb2
from sandbox.tasklet.sidecars.resource_manager.proto import resource_manager_api_pb2
from sandbox.tasklet.sidecars.resource_manager.proto import resource_manager_api_pb2_grpc

from tasklet.api.v2 import context_pb2

from tasklet.experimental.sdk.py.dummy import interface as dummy_sdk
from tasklet.experimental.examples.proto import sb_resource_pb2


class ResourceException(Exception):
    """ Class for errors with resources """


def main():
    tasklet_io = dummy_sdk.TaskletInterface()

    request: sb_resource_pb2.SbResourceInput = tasklet_io.read_input(sb_resource_pb2.SbResourceInput())
    resource_type: str = request.type
    resource_owner: str = request.owner

    context: context_pb2.Context = tasklet_io.get_context()

    rm_address: str = context.environment.sandbox_resource_manager.address
    resource_manager = resource_manager_api_pb2_grpc.ResourceManagerAPIStub(grpc.insecure_channel(rm_address))
    print("Created grpc client for sandbox resource manager to address {}.".format(rm_address))

    print("Searching resource with type '{}' and owner '{}'.".format(resource_type, resource_owner))
    resources: resource_manager_api_pb2.GetResourcesResponse = resource_manager.GetResources(
        resource_manager_api_pb2.GetResourcesRequest(type=resource_type, owner=resource_owner, state="READY")
    )
    print("Found {} resources.".format(len(resources.resources)))

    if len(resources.resources) == 0:
        raise ResourceException(
            "Resource with type '{}' and owner '{}' not found.".format(resource_type, resource_owner)
        )

    resource: resource_pb2.Resource = resources.resources[0]

    print("Download resource {}.".format(resource.id))

    resource_path: str = resource_manager.DownloadResource(
        resource_manager_api_pb2.DownloadResourceRequest(id=resource.id)
    ).path

    print("Resource {} downloaded to '{}'.".format(resource.id, resource_path))
    if not os.path.exists(resource_path):
        raise ResourceException("Resource not found in path '{}'".format(resource_path))

    new_resource_path: str = os.path.abspath("test_resource")
    print("Copy resource data from '{}' to '{}'".format(resource_path, new_resource_path))
    if os.path.isfile(resource_path):
        shutil.copy(resource_path, new_resource_path)
    else:
        shutil.copytree(resource_path, new_resource_path)

    print("Create new sandbox resource with the same data")

    new_resource: resource_pb2.Resource = resource_manager.CreateResource(
        resource_manager_api_pb2.CreateResourceRequest(
            type=resource_type, owner="TASKLETS", attributes={"test_attr": "1"}, path=new_resource_path,
            description="Sample resource for testing tasklets",
        )
    ).resource

    print("Created resource with id {}".format(new_resource.id))

    response = sb_resource_pb2.SbResourceOutput(id=new_resource.id)

    tasklet_io.write_output(response)


if __name__ == "__main__":
    main()
