package access

import (
	"context"
	"strings"
	"testing"

	"a.yandex-team.ru/tasklet/api/v2"
	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	stafftestutils "a.yandex-team.ru/tasklet/experimental/internal/yandex/staff/testutils"
	"github.com/stretchr/testify/assert"
)

func TestCheckPermission(t *testing.T) {
	og := testutils.NewObjectGenerator()
	namespaces, tasklets := og.CreateTestObjectsInMemory()
	nsMap := map[string]*taskletv2.Namespace{}
	tlMap := map[string]*taskletv2.Tasklet{}
	for _, ns := range namespaces {
		nsMap[ns.Meta.Name] = ns
	}
	for _, tl := range tasklets {
		tlMap[tl.Meta.Name] = tl
	}
	logger := testutils.TwistMakeLogger(testutils.TwistTmpDir(t), strings.ReplaceAll(t.Name(), "/", "_")+".log")
	staffClient := stafftestutils.StaffClientMock{Logger: logger}
	staffGroupsCache := staff.NewStaffGroupsCache(&staffClient)
	defer staffGroupsCache.Stop()
	pc := PermissionsChecker{
		logger:     logger,
		staffCache: staffGroupsCache,
	}
	namespaceRoles := []acmodel.BaseAccessMaskType{
		acmodel.BaseNamespaceRead,
		acmodel.BaseCreateTasklet,
		acmodel.BaseNamespaceOwner,
	}
	taskletRoles := []acmodel.BaseAccessMaskType{
		acmodel.BaseTaskletRead,
		acmodel.BaseTaskletWrite,
		acmodel.BaseTaskletRun,
		acmodel.BaseTaskletOwner,
	}
	ctx := context.Background()

	// Test ns owner permissions
	for _, role := range namespaceRoles {
		assert.NoError(
			t, pc.CheckPermissions(
				ctx, testutils.UserBilly, role, &acmodel.AccessData{
					Namespace: nsMap[testutils.NamespaceArcadia],
				},
			),
		)
	}
	// Test tasklet owner permissions
	for _, role := range taskletRoles {
		assert.NoError(
			t, pc.CheckPermissions(
				ctx, testutils.UserBilly, role, &acmodel.AccessData{
					Namespace: nsMap[testutils.NamespaceArcadia],
					Tasklet:   tlMap[testutils.TaskletTestTasklet],
				},
			),
		)
	}
	// Test user read tasklet permissions if hi has namespace read permissions
	for _, tl := range tasklets {
		assert.NoError(
			t, pc.CheckPermissions(
				ctx, testutils.UserBilly, acmodel.BaseTaskletRead, &acmodel.AccessData{
					Namespace: nsMap[tl.Meta.Namespace],
					Tasklet:   tl,
				},
			),
		)
	}
	// Test user with write permissions has them
	assert.NoError(
		t, pc.CheckPermissions(
			ctx, testutils.UserMark, acmodel.BaseTaskletWrite, &acmodel.AccessData{
				Namespace: nsMap[testutils.NamespaceArcadia],
				Tasklet:   tlMap[testutils.TaskletTestTasklet],
			},
		),
	)
	// Test user hasn't run permission
	assert.Error(
		t, pc.CheckPermissions(
			ctx, testutils.UserMark, acmodel.BaseTaskletRun, &acmodel.AccessData{
				Namespace: nsMap[testutils.NamespaceArcadia],
				Tasklet:   tlMap[testutils.TaskletTestTasklet],
			},
		),
	)
	// Test user in group has it permissions
	assert.NoError(
		t, pc.CheckPermissions(
			ctx, testutils.UserBilly, acmodel.BaseTaskletOwner, &acmodel.AccessData{
				Namespace: nsMap[testutils.NamespaceTaxi],
				Tasklet:   tlMap[testutils.TaskletTestTaxiTasklet],
			},
		),
	)
}
