package apiclient

import (
	"context"
	"strings"

	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/requestctx"
	"a.yandex-team.ru/tasklet/experimental/internal/xgrpc"
)

// InsecureCredentials acts like oauth2.oauthAccess, but doesn't require secure transport
// TODO: Use oauth2.NewOauthAccess after ssl activation
type InsecureCredentials struct {
	token  string
	method consts.CredentialsMethod
}

func NewCredentials(token string, method consts.CredentialsMethod) *InsecureCredentials {
	return &InsecureCredentials{token: strings.TrimSpace(token), method: method}
}

func (ic *InsecureCredentials) GetRequestMetadata(_ context.Context, _ ...string) (map[string]string, error) {
	return map[string]string{
		consts.AuthorizationHeader: string(ic.method) + " " + ic.token,
	}, nil
}

func (ic *InsecureCredentials) RequireTransportSecurity() bool {
	return false
}

type InsecureTestCredentials struct {
	kind        requestctx.SubjectKind
	user        string
	executionID consts.ExecutionID
}

func NewInsecureTestCredentials(
	kind requestctx.SubjectKind,
	user string,
	executionID consts.ExecutionID,
) *InsecureTestCredentials {
	return &InsecureTestCredentials{
		kind:        kind,
		user:        user,
		executionID: executionID,
	}
}

func (ic *InsecureTestCredentials) GetRequestMetadata(_ context.Context, _ ...string) (map[string]string, error) {
	rv := make(map[string]string, 1)
	switch ic.kind {
	case requestctx.UserSubject:
		rv[xgrpc.TestUserMetadataKey] = ic.user
	case requestctx.ExecutionSubject:
		rv[xgrpc.TestExecutionMetadataKey] = ic.executionID.String()
	default:
		panic(ic.kind)
	}
	return rv, nil
}

func (ic *InsecureTestCredentials) RequireTransportSecurity() bool {
	return false
}
