package apiclient

import (
	"context"

	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials"
	"google.golang.org/grpc/credentials/insecure"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	ypsdresolver "a.yandex-team.ru/infra/yp_service_discovery/golang/wrapper/grpcresolver"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/priv/v1"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
)

// NotSecureOAuthCredentials acts like oauth2.oauthAccess, but doesn't require secure transport
// TODO: Use oauth2.NewOauthAccess after ssl activation
type NotSecureOAuthCredentials struct {
	Token string
}

func (oc *NotSecureOAuthCredentials) GetRequestMetadata(_ context.Context, _ ...string) (map[string]string, error) {
	return map[string]string{
		consts.AuthorizationHeader: string(consts.OAuthMethod) + " " + oc.Token,
	}, nil
}

func (oc *NotSecureOAuthCredentials) RequireTransportSecurity() bool {
	return false
}

type connectionFactory func() (*grpc.ClientConn, error)

func newConnFactory(conf *Config, auth credentials.PerRPCCredentials, logger log.Logger) (connectionFactory, error) {
	var address = ""
	if conf.EndpointSetName != "" {
		address = ypsdresolver.GrpcTarget(conf.EndpointSetName, resolver.AvailableClusters...)
	} else {
		address = conf.EndpointAddress
	}

	logger.Infof("Creating grpc-client for %q", address)

	options := []grpc.DialOption{
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithUserAgent("tasklet-executor"),
		grpc.WithDefaultServiceConfig(`{"loadBalancingPolicy":"round_robin"}`),
		grpc.WithDisableServiceConfig(),
		grpc.WithResolvers(ypsdresolver.NewBuilder()),
	}
	options = append(options, grpc.WithPerRPCCredentials(auth))

	return func() (*grpc.ClientConn, error) {
		return grpc.Dial(address, options...)
	}, nil
}

func newTaskletAPIClient(factory connectionFactory) (taskletv2.TaskletServiceClient, error) {
	conn, err := factory()
	if err == nil {
		return taskletv2.NewTaskletServiceClient(conn), nil
	} else {
		return nil, err
	}
}

func newInternalAPIClient(factory connectionFactory) (privatetaskletv1.InternalServiceClient, error) {
	conn, err := factory()
	if err == nil {
		return privatetaskletv1.NewInternalServiceClient(conn), nil
	} else {
		return nil, err
	}
}

func newSchemaRegistryClient(factory connectionFactory) (taskletv2.SchemaRegistryServiceClient, error) {
	conn, err := factory()
	if err == nil {
		return taskletv2.NewSchemaRegistryServiceClient(conn), nil
	} else {
		return nil, err
	}
}

type ServerConnection struct {
	l               log.Logger
	SchemaRegistry  taskletv2.SchemaRegistryServiceClient
	InternalService privatetaskletv1.InternalServiceClient
	PublicService   taskletv2.TaskletServiceClient
}

func NewServerConnection(logger log.Logger, conf *Config, auth credentials.PerRPCCredentials) (
	*ServerConnection,
	error,
) {
	rv := &ServerConnection{
		l: logger,
	}

	if connectionMaker, err := newConnFactory(conf, auth, logger); err != nil {
		return nil, err
	} else {
		if apiClient, err := newInternalAPIClient(connectionMaker); err != nil {
			return nil, xerrors.Errorf("Failed to initialize private API client: %w", err)
		} else {
			rv.InternalService = apiClient
		}
		if apiClient, err := newSchemaRegistryClient(connectionMaker); err != nil {
			return nil, xerrors.Errorf("Failed to initialize schema registry API client: %w", err)
		} else {
			rv.SchemaRegistry = apiClient
		}
		if apiClient, err := newTaskletAPIClient(connectionMaker); err != nil {
			return nil, xerrors.Errorf("Failed to initialize schema registry API client: %w", err)
		} else {
			rv.PublicService = apiClient
		}

	}

	return rv, nil
}
