package apiserver

import (
	"fmt"
	"net"

	grpcMiddleware "github.com/grpc-ecosystem/go-grpc-middleware"
	grpcZap "github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	grpcRecovery "github.com/grpc-ecosystem/go-grpc-middleware/recovery"
	uberZap "go.uber.org/zap"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials"
	"google.golang.org/grpc/reflection"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/tasklet/api/priv/v1"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/xgrpc"
)

type Server struct {
	conf   *Config
	grpc   *grpc.Server
	logger log.Logger
}

func New(
	c *Config,
	l log.Logger,
	publicAPIHandler taskletv2.TaskletServiceServer,
	internalAPIHandler privatetaskletv1.InternalServiceServer,
	schemaRegistry taskletv2.SchemaRegistryServiceServer,
	sandboxSessionChecker SandboxSessionChecker,
	mr metrics.Registry,
) (*Server, error) {

	_ = mr

	var zapOpts []grpcZap.Option
	grpcServerLog, ok := l.WithName("grpc_server").(*zap.Logger)
	if !ok {
		panic("Failed to get logger")
	}

	mw, err := NewMiddleware(c.Middleware, l.WithName("middleware"), sandboxSessionChecker)
	if err != nil {
		return nil, err
	}

	serverOptions := make([]grpc.ServerOption, 0)
	if c.EnableTLS {
		serverTLS, err := credentials.NewServerTLSFromFile(c.CertFile, c.KeyFile)
		if err != nil {
			l.Errorf("Failed to generate credentials %v", err)
			return nil, err
		}
		serverOptions = append(serverOptions, grpc.Creds(serverTLS))
	}

	// Unary interceptor
	serverOptions = append(
		serverOptions,
		grpcMiddleware.WithUnaryServerChain(
			// LogMetadata(grpcServerLog),
			grpcRecovery.UnaryServerInterceptor(
				grpcRecovery.WithRecoveryHandlerContext(
					xgrpc.PanicHandler(
						l,
						c.ExposeError,
					),
				),
			),
			UnaryRequestIDGenerator(l.WithName("request_id_middleware")),
			grpcZap.UnaryServerInterceptor(grpcServerLog.L.WithOptions(uberZap.AddCallerSkip(1)), zapOpts...),
			mw.UnaryInterceptor,
		),
	)

	// StreamInterceptor
	serverOptions = append(
		serverOptions,
		grpcMiddleware.WithStreamServerChain(
			grpcRecovery.StreamServerInterceptor(
				grpcRecovery.WithRecoveryHandlerContext(
					xgrpc.PanicHandler(
						l,
						c.ExposeError,
					),
				),
			),
			grpcZap.StreamServerInterceptor(grpcServerLog.L.WithOptions(uberZap.AddCallerSkip(1)), zapOpts...),
		),
	)

	grpcServer := grpc.NewServer(serverOptions...)

	taskletv2.RegisterTaskletServiceServer(
		grpcServer,
		publicAPIHandler,
	)
	privatetaskletv1.RegisterInternalServiceServer(
		grpcServer,
		internalAPIHandler,
	)
	taskletv2.RegisterSchemaRegistryServiceServer(
		grpcServer,
		schemaRegistry,
	)
	reflection.Register(grpcServer)

	return &Server{c, grpcServer, l}, nil
}

func (s *Server) ListenAndServe() error {
	hostPort := fmt.Sprintf("[::]:%v", s.conf.Port)
	s.logger.Infof("listening on %q", hostPort)
	l, err := net.Listen("tcp", hostPort)
	if err != nil {
		return err
	}
	return s.grpc.Serve(l)
}

func (s *Server) Shutdown() {
	s.grpc.Stop()
}
