package v2

import (
	"context"

	ydbOptions "github.com/ydb-platform/ydb-go-sdk/v3/table/options"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/tasklet/experimental/internal/ydbmigrate"
)

// Migration info:
//
// Tables: executions, executions_archive
//
// Rename columns:
//  	execution_id -> monotonic_id
// Rename secondary indexes:
//		"tasklet_id_execution_id_view" -> tasklet_id_monotonic_id_view
// 		"build_id_execution_id_view" -> build_id_monotonic_id_view

// TaskletIDColumn            = "tasklet_id"
// BuildIDColumn              = "build_id"

// language=YQL
var addIndexQueries = []string{
	"ALTER TABLE `executions` ADD INDEX `tasklet_id_monotonic_id_view` GLOBAL ON (`tasklet_id`, `monotonic_id`);",
	"ALTER TABLE `executions` ADD INDEX `build_id_monotonic_id_view` GLOBAL ON (`build_id`, `monotonic_id`);",
	"ALTER TABLE `executions_archive` ADD INDEX `tasklet_id_monotonic_id_view` GLOBAL ON (`tasklet_id`, `monotonic_id`);",
	"ALTER TABLE `executions_archive` ADD INDEX `build_id_monotonic_id_view` GLOBAL ON (`build_id`, `monotonic_id`);",
}

// language=YQL
var copyQueries = []string{
	"UPDATE `executions` SET `monotonic_id` = `execution_id`;",
	"UPDATE `executions_archive` SET `monotonic_id` = `execution_id`;",
}

// language=YQL
var dropIndexQueries = []string{
	"ALTER TABLE `executions` DROP INDEX `tasklet_id_execution_id_view`;",
	"ALTER TABLE `executions` DROP INDEX `build_id_execution_id_view`;",
	"ALTER TABLE `executions_archive` DROP INDEX `tasklet_id_execution_id_view`;",
	"ALTER TABLE `executions_archive` DROP INDEX `build_id_execution_id_view`;",
}

func renameExecutionIDColumn(ctx context.Context, cli *xydb.Client) error {

	// NB: add new tables
	if err := cli.AlterTable(
		ctx,
		"executions",
		ydbOptions.WithAddColumn("monotonic_id", ydbTypes.Optional(ydbTypes.TypeInt64)),
	); err != nil {
		return err
	}
	if err := cli.AlterTable(
		ctx,
		"executions_archive",
		ydbOptions.WithAddColumn("monotonic_id", ydbTypes.Optional(ydbTypes.TypeInt64)),
	); err != nil {
		return err
	}

	// NB: add new indexes
	for _, q := range addIndexQueries {
		if err := cli.SchemeQuery(ctx, q); err != nil {
			return xerrors.Errorf("query failed. Query: %q, error: %w", q, err)
		}
	}

	for _, q := range copyQueries {
		if _, err := cli.Do(ctx, q, cli.WriteTxControl); err != nil {
			return xerrors.Errorf("query failed. Query: %q, error: %w", q, err)
		}
	}

	for _, q := range dropIndexQueries {
		if err := cli.SchemeQuery(ctx, q); err != nil {
			return xerrors.Errorf("query failed. Query: %q, error: %w", q, err)
		}
	}

	if err := cli.AlterTable(
		ctx,
		"executions",
		ydbOptions.WithDropColumn("execution_id"),
	); err != nil {
		return err
	}
	if err := cli.AlterTable(
		ctx,
		"executions_archive",
		ydbOptions.WithDropColumn("execution_id"),
	); err != nil {
		return err
	}

	return nil
}

var MigrateV1V2 = ydbmigrate.Migration{
	BaseVersion: 1,
	Description: "rename execution monotonic id column",
	Mutations: []ydbmigrate.Mutation{
		{
			MutationName: "0000_rename_execution_id_column",
			Func:         renameExecutionIDColumn,
		},
	},
}
