package v3

import (
	"context"
	"fmt"

	ydbTable "github.com/ydb-platform/ydb-go-sdk/v3/table"
	ydbNamed "github.com/ydb-platform/ydb-go-sdk/v3/table/result/named"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"
	"go.uber.org/zap/zapcore"
	"google.golang.org/protobuf/encoding/protojson"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/cmd/server/migrations/shared"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/tasklet/experimental/internal/ydbmigrate"
)

// language=YQL
const errorAndResultCheckQuery = `
PRAGMA TablePathPrefix("/ru-prestable/ydb_home/alximik/alximik/dbv1");
SELECT
    a.id, CAST(JSON_VALUE(a.p, "$.status.error.description") AS String) as error, b.output
FROM executions_archive AS a
INNER JOIN execution_blobs AS b
ON a.id == b.id
WHERE
  (
      (output IS NULL OR UNWRAP(output) == "\u0000")  AND NOT JSON_EXISTS(a.p, "$.status.error")

  ) OR (
      (output IS NOT NULL AND UNWRAP(output) != "\u0000")  AND JSON_EXISTS(a.p, "$.status.error")
  )
`

// NB: interactive check
var _ = errorAndResultCheckQuery

// language=YQL
const deleteQueryTemplate = `
DECLARE $id AS String; -- execution id
DELETE FROM %s WHERE id = $id;
DELETE FROM execution_blobs WHERE id = $id;
`

// language=YQL
const updateQueryTemplate = `
DECLARE $id AS String;
DECLARE $p as JsonDocument;
DISCARD SELECT
    ENSURE(0, output IS NULL OR UNWRAP(output) == "\u0000", "BAD ID: " || $id)
FROM execution_blobs
WHERE id = $id;

UPDATE %s SET p = $p WHERE id = $id;
UPDATE execution_blobs SET output = NULL WHERE id = $id;
`

type Consumer struct {
	logger             log.Logger
	processedRowsCount int
	updatedExecutions  map[consts.ExecutionID][]byte
	droppedExecutions  []consts.ExecutionID
	rowExecutionID     consts.ExecutionID
	rowPayload         []byte
}

func (c *Consumer) GetColumnSet() []string {
	return []string{"id", "p"}
}

func (c *Consumer) NextReadValues() xydb.ReadValues {
	return xydb.ReadValues{
		ydbNamed.OptionalWithDefault("id", (*string)(&c.rowExecutionID)),
		ydbNamed.OptionalWithDefault("p", &c.rowPayload),
	}
}

var unmarshaller = protojson.UnmarshalOptions{DiscardUnknown: false, AllowPartial: false}

func (c *Consumer) ConsumeReadValues() error {
	c.processedRowsCount += 1
	ex := &taskletv2.Execution{}
	originalPayload := c.rowPayload
	if err := unmarshaller.Unmarshal(c.rowPayload, ex); err != nil {
		c.logger.Warnf("Dropping exectuion. ID: %q, Err: %v", c.rowExecutionID, err)
		c.droppedExecutions = append(c.droppedExecutions, c.rowExecutionID)
		return nil
	}
	if consts.ExecutionID(ex.Meta.Id) != c.rowExecutionID {
		return xerrors.Errorf("unexpected: PK: %q, Meta.ID: %q", c.rowExecutionID, ex.Meta.Id)
	}

	if ex.Status.GetError().GetDescription() == "" {
		if ex.Status.GetError() != nil {
			c.logger.Infof("Empty error. ExecutionID: %v", c.rowExecutionID)
		}
		bytes, err := protojson.Marshal(ex)
		if err != nil {
			return xerrors.Errorf("marshall failed. ExecutionID: %q, error: %w", c.rowExecutionID, err)
		}
		diff := shared.MustDiffJSON(originalPayload, bytes)
		if diff == "" {
			c.logger.Infof("Skip ExecutionID: %v", c.rowExecutionID)
			return nil
		}
		c.updatedExecutions[c.rowExecutionID] = bytes
		c.logger.Infof("====== %q\n%s", c.rowExecutionID, diff)
		panic(c.rowExecutionID)
	}

	ex.Status.ProcessingResult = &taskletv2.ProcessingResult{
		Kind: &taskletv2.ProcessingResult_ServerError{
			ServerError: &taskletv2.ServerError{
				Code:        taskletv2.ErrorCodes_ERROR_CODE_GENERIC,
				Description: ex.Status.Error.Description,
				IsTransient: false,
			},
		},
	}
	ex.Status.Error = nil
	if ex.Status.Result != nil {
		return xerrors.Errorf("non nil result: ExecutionID: %q", c.rowExecutionID)
	}
	bytes, err := protojson.Marshal(ex)
	if err != nil {
		return xerrors.Errorf("marshall failed. ExecutionID: %q, error: %w", c.rowExecutionID, err)
	}
	c.updatedExecutions[c.rowExecutionID] = bytes
	diff := shared.MustDiffJSON(originalPayload, bytes)
	c.logger.Infof("====== %q\n%s", c.rowExecutionID, diff)
	return nil
}

func (c *Consumer) Done() {
	// noop
}

func do(ctx context.Context, cli *xydb.Client, table string) error {
	// OutputFormat
	// {
	//  "status": {
	//    "processingResult": {
	//      "serverError": {
	//        "code": "ERROR_CODE_GENERIC",
	//        "description": "boo"
	//      }
	//    }
	//  }
	// }
	logConf := zap.ConsoleConfig(log.DebugLevel)
	logConf.EncoderConfig.EncodeLevel = zapcore.CapitalColorLevelEncoder
	logger := zap.Must(logConf)

	consumer := &Consumer{
		logger:            logger,
		updatedExecutions: make(map[consts.ExecutionID][]byte),
		rowExecutionID:    "",
		rowPayload:        nil,
	}
	if err := cli.StreamReadTable(ctx, table, consumer); err != nil {
		return xerrors.Errorf("stream read table failed: %w", err)
	}
	logger.Infof("ProcessedRowsCount: %d", consumer.processedRowsCount)
	logger.Infof("UpdatedRowsCount: %d", len(consumer.updatedExecutions))
	logger.Infof("Dropped executions count: %d", len(consumer.droppedExecutions))
	query := fmt.Sprintf(deleteQueryTemplate, table)
	for _, executionID := range consumer.droppedExecutions {
		logger.Infof("Deleting execution. ID: %q", executionID)
		res, err := cli.Do(
			ctx,
			query,
			cli.WriteTxControl,
			ydbTable.ValueParam("$id", ydbTypes.StringValueFromString(executionID.String())),
		)
		if err != nil {
			return xerrors.Errorf("Failed to delete. ExecutionID: %q,  Err: %v", executionID, err)
		}
		_ = res.Close()
	}

	query = fmt.Sprintf(updateQueryTemplate, table)
	for executionID, payload := range consumer.updatedExecutions {
		res, err := cli.Do(
			ctx,
			query,
			cli.WriteTxControl,
			ydbTable.ValueParam("$id", ydbTypes.StringValueFromString(executionID.String())),
			ydbTable.ValueParam("$p", ydbTypes.JSONDocumentValueFromBytes(payload)),
		)
		if err != nil {
			return xerrors.Errorf("Failed to update. ExecutionID: %q,  Err: %v", executionID, err)
		}
		_ = res.Close()
	}
	return nil
}

// https://a.yandex-team.ru/review/2629273/files/2#file-0-189853196:R370
// Summary:
//   1. Renamed ExecutionResult -> ExecutionOutput   -- does not change json representation
//   2. deprecated status.error & status.result.
//       * status.result is stripped from json and stored in separate table. Not affecting representation, storage
//         will restore document in new form
//       * status.error must be stripped and converted to status.processing_result.server_error
//    Also, check for error & result mutual exclusion
//
//
//   execution_journal table stores data temporary and may be not patched

var MigrateV2V3 = ydbmigrate.Migration{
	BaseVersion: 2,
	Description: "merge result and error to processing_result",
	Mutations: []ydbmigrate.Mutation{
		{
			MutationName: "0000_process_executions_archive",
			Func: func(ctx context.Context, cli *xydb.Client) error {
				return do(
					ctx,
					cli,
					"executions_archive",
				)
			},
		},
		{
			MutationName: "0001_process_executions",
			Func:         func(ctx context.Context, cli *xydb.Client) error { return do(ctx, cli, "executions") },
		},
	},
}
