package server

import (
	"context"
	"net/http"
	"os"
	"os/signal"
	"runtime/debug"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/tasklet/experimental/internal/access"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/staff"
	grpcZap "github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics/collect"
	"a.yandex-team.ru/library/go/core/metrics/collect/policy/inflight"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/apiserver"
	"a.yandex-team.ru/tasklet/experimental/internal/handler"
	"a.yandex-team.ru/tasklet/experimental/internal/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/locks"
	"a.yandex-team.ru/tasklet/experimental/internal/processor"
	"a.yandex-team.ru/tasklet/experimental/internal/revproxy"
	"a.yandex-team.ru/tasklet/experimental/internal/sbresolver"
	"a.yandex-team.ru/tasklet/experimental/internal/schemaregistry"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/ydbstore"
	"a.yandex-team.ru/tasklet/experimental/internal/utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/ytdriver"
	"a.yandex-team.ru/tasklet/experimental/internal/ydbmigrate"
)

// SandboxSessionCheckerImpl FIXME: move this interop to proper location
type SandboxSessionCheckerImpl struct {
	sbx *sandbox.Client
}

func (s SandboxSessionCheckerImpl) CheckSandboxSession(
	ctx context.Context,
	session sandbox.SandboxSession,
) (sandbox.SessionInfo, error) {
	return s.sbx.SearchSession(ctx, session)
}

func (s SandboxSessionCheckerImpl) CheckExternalSession(ctx context.Context, session sandbox.SandboxExternalSession) (
	sandbox.ExternalSessionInfo,
	error,
) {
	return s.sbx.GetExternalSession(ctx, session)
}

var _ apiserver.SandboxSessionChecker = (*SandboxSessionCheckerImpl)(nil)

func serverCmdHandler(_ *cobra.Command, _ []string) (err error) {
	var loggers lib.Loggers
	loggers, err = lib.NewLoggers(conf.Logging)
	if err != nil {
		return
	}
	coreLog := loggers[lib.CoreLogger]

	defer func() {
		if r := recover(); r != nil {
			err = xerrors.Errorf("Panic: %+v", r)
			stack := debug.Stack()
			fields := []log.Field{
				log.Error(err),
				log.Any("trace", string(stack)),
			}
			coreLog.Error("Recovered panic", fields...)
		}
	}()

	err = doRunServer(loggers)
	if err != nil {
		coreLog.Error("server crashed", log.Error(err))
	}
	return
}

func doRunServer(loggers lib.Loggers) error {

	{
		grpcLog := loggers[lib.GRPCLogger]
		if zaplog, ok := grpcLog.(*zap.Logger); ok {
			grpcZap.ReplaceGrpcLoggerV2(zaplog.L)
		}
	}
	rootCtx, rootCancel := context.WithCancel(context.Background())
	defer rootCancel()
	coreLog := loggers[lib.CoreLogger]

	solomonRegistry := solomon.NewRegistry(
		solomon.NewRegistryOpts().
			SetRated(true).
			AddCollectors(
				rootCtx,
				inflight.NewCollectorPolicy(),
				collect.GoMetrics,
				collect.ProcessMetrics,
			),
	)

	ydbClient, err := xydb.NewClient(
		rootCtx,
		conf.DB.Ydb,
		utils.MustToken(utils.LoadToken(conf.DB.Ydb.TokenPath)),
		coreLog.WithName("ydb_client"),
		solomonRegistry,
	)
	if err != nil {
		coreLog.Error("Failed to construct YDB client", log.Error(err))
		return err
	}

	defer func() {
		err := ydbClient.Close(rootCtx)
		if err != nil {
			coreLog.Warn("ydb close error", log.Error(err))
		}
	}()

	// NB: ensure database version

	mc := ydbmigrate.NewMigrationClient(ydbClient, coreLog)
	migrateErr := mc.MigrationCheckLoop(rootCtx, ydbstore.SchemaVersion)

	if migrateErr != nil {
		coreLog.Error("Error on YDB migration check", log.Error(migrateErr))
		return migrateErr
	}

	var db storage.IStorage = ydbstore.NewStorage(ydbClient, coreLog.WithName("storage"))

	sandboxToken := ""
	if conf.Sandbox.TokenPath != "" {
		rv, err := utils.LoadToken(conf.Sandbox.TokenPath)
		if err != nil {
			return xerrors.Errorf("Failed to load sandbox token: %w", err)
		}
		sandboxToken = rv
	}
	sbx, err := sandbox.New(
		conf.Sandbox, sandboxToken, coreLog.WithName("sandbox"), solomonRegistry.WithPrefix("sandbox"),
	)
	if err != nil {
		return err
	}

	sbxGroupCache := sandbox.NewGroupCache(conf.GroupCacheConfig, sbx, coreLog.WithName("sandbox_group_cache"))
	defer sbxGroupCache.Stop()

	var staffClient *staff.Client

	if conf.Staff.Enabled {
		tvmClient, tvmErr := tvmtool.NewDeployClient(
			tvmtool.WithCacheEnabled(true),
			tvmtool.WithLogger(coreLog.WithName("tvmtool").Structured()),
		)
		if tvmErr != nil {
			coreLog.Error("Unable to create TVM Client", log.Error(tvmErr))
			return err
		}

		staffClient = &staff.Client{
			Logger:      coreLog.WithName("staff_client"),
			TVMClient:   tvmClient,
			TVMClientID: conf.Staff.TVM,
			URL:         conf.Staff.URL,
			HTTPClient:  &http.Client{},
		}
	}
	staffCache := staff.NewStaffGroupsCache(staffClient)
	staffCache.Enabled = conf.Staff.Enabled
	defer staffCache.Stop()
	permissionsChecker := access.NewAccessChecker(
		staffCache,
		coreLog.WithName("access_checker"),
	)

	apiHandler, err := handler.New(
		coreLog.WithName("handler"),
		conf.Handler,
		db,
		sbxGroupCache,
		sbx,
		permissionsChecker,
	)
	if err != nil {
		return err
	}

	schemaRegistryHandler, err := schemaregistry.New(
		coreLog.WithName("schema_registry"),
		db,
	)
	if err != nil {
		return err
	}
	grpcServer, err := apiserver.New(
		conf.APIServer,
		loggers[lib.GRPCLogger],
		apiHandler,
		apiHandler,
		schemaRegistryHandler,
		SandboxSessionCheckerImpl{sbx: sbx},
		solomonRegistry.WithPrefix("api"),
	)

	if err != nil {
		return err
	}

	httpProxy, err := revproxy.NewProxy(conf.Proxy, loggers[lib.HTTPLogger], solomonRegistry)
	if err != nil {
		return err
	}

	var ytc *ytdriver.YTDriver
	if conf.YTConfig.Enabled {
		ytDriver, err := ytdriver.New(conf.YTConfig, coreLog.WithName("yt_driver"), sbx, conf.Executor)
		if err != nil {
			coreLog.Errorf("Failed to initialize YT driver: %+v", err)
		}
		ytc = ytDriver
	}

	distributedLocksRepo := locks.NewLocksRepo(ydbClient)

	executionProcessor, err := processor.New(
		conf.Processor,
		conf.Executor,
		coreLog.WithName("processor"),
		db,
		sbx,
		ytc,
		locks.NewLocker(distributedLocksRepo, "processor", conf.Installation, coreLog.WithName("processor_locker")),
		solomonRegistry.WithPrefix("processor"),
	)
	if err != nil {
		return err
	}

	resResolver, err := sbresolver.New(conf.ResourceResolver, sbx, coreLog.WithName("sbresolver"))
	if err != nil {
		return err
	}
	if err := resResolver.Bootstrap(rootCtx); err != nil {
		coreLog.Error("Sandbox resource resolver bootstrap failed", log.Error(err))
		return xerrors.Errorf("resource resolver bootstrap failed: %w", err)
	}

	// Starting components
	errorChan := make(chan error, 4)
	stop := make(chan os.Signal, 1)
	signal.Notify(stop, syscall.SIGINT, syscall.SIGTERM)

	go func() {
		coreLog.Infof("Starting gRPC server")
		err := grpcServer.ListenAndServe()
		coreLog.Infof("gRPC server stopped: %v", err)
		errorChan <- err
	}()

	go func() {
		// NB: let grpc server initialize
		<-time.After(1 * time.Second)
		coreLog.Infof("Starting http proxy")
		err := httpProxy.Serve()
		coreLog.Infof("Http proxy stopped: %v", err)
		errorChan <- err
	}()

	go func() {
		// NB: let grpc server initialize
		<-time.After(1 * time.Second)
		coreLog.Infof("Starting processor")
		err := executionProcessor.Serve()
		coreLog.Infof("Processor stopped: %v", err)
		errorChan <- err
	}()

	go func() {
		coreLog.Infof("Starting sbresolver")
		err := resResolver.Run()
		coreLog.Infof("sbresolver stopped: %v", err)
		errorChan <- err
	}()

	select {
	case <-stop:
		// NB: graceful stop
		ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
		defer cancel()

		resResolver.Stop()
		executionProcessor.Stop()
		if proxyErr := httpProxy.Stop(ctx); proxyErr != nil {
			return proxyErr
		}
		grpcServer.Shutdown()
	case err := <-errorChan:
		// NB: emergency stop
		return err
	}
	return nil
}

var serverCmd = &cobra.Command{
	Use:     "run",
	Short:   "Run server",
	PreRunE: validateConfig,
	RunE:    serverCmdHandler,
}
