package server

import (
	"context"
	"fmt"
	"os"
	"time"

	"github.com/gofrs/uuid"
	"github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	"github.com/spf13/cobra"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics/nop"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/ydbstore"
	"a.yandex-team.ru/tasklet/experimental/internal/utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/ytdriver"
)

func ytRunHandler(_ *cobra.Command, _ []string) error {
	loggers, err := lib.NewLoggers(conf.Logging)
	if err != nil {
		return err
	}

	{
		grpcLog := loggers[lib.GRPCLogger]
		if zaplog, ok := grpcLog.(*zap.Logger); ok {
			grpc_zap.ReplaceGrpcLoggerV2(zaplog.L)
		}
	}
	rootCtx, rootCancel := context.WithCancel(context.Background())
	_ = rootCtx
	defer rootCancel()
	coreLog := loggers[lib.CoreLogger]

	sandboxToken, err := utils.LoadToken(conf.Sandbox.TokenPath)
	if err != nil {
		return err
	}
	sbx, err := sandbox.New(conf.Sandbox, sandboxToken, coreLog.WithName("sandbox"), &nop.Registry{})
	if err != nil {
		coreLog.Errorf("Failed to initialize SBX driver: %+v", err)
		return err
	}

	ytd, err := ytdriver.New(conf.YTConfig, coreLog.WithName("yt_driver"), sbx, conf.Executor)
	if err != nil {
		coreLog.Errorf("Failed to initialize YT driver: %+v", err)
		return err
	}

	_ = ytd
	defer ytd.Stop()

	ydbClient, err := xydb.NewClient(
		rootCtx,
		conf.DB.Ydb,
		utils.MustToken(utils.LoadToken(conf.DB.Ydb.TokenPath)),
		coreLog.WithName("ydb_client"),
		nil,
	)
	if err != nil {
		coreLog.Error("Failed to construct YDB client", log.Error(err))
		return err
	}

	defer ydbClient.Close(rootCtx)
	var db storage.IStorage = ydbstore.NewStorage(ydbClient, coreLog.WithName("storage"))

	bb, err := db.GetBuild(rootCtx, buildID)
	if err != nil {
		return err
	}
	fmt.Println(protojson.Format(bb))

	ex := &taskletv2.Execution{
		Meta: &taskletv2.ExecutionMeta{
			Id:        uuid.Must(uuid.NewV4()).String(),
			TaskletId: bb.Meta.TaskletId,
			BuildId:   buildID,
			CreatedAt: timestamppb.New(time.Now()),
		},
		Spec: &taskletv2.ExecutionSpec{
			Author:          "",
			ReferencedLabel: "",
			Requirements: &taskletv2.ExecutionRequirements{
				AccountId: "",
			},
			Input: &taskletv2.ExecutionInput{},
		},
		Status: &taskletv2.ExecutionStatus{
			Status:    0,
			Result:    nil,
			Error:     nil,
			Processor: nil,
		},
	}
	if data, err := os.ReadFile(inputFile); err != nil {
		return err
	} else {
		msg := &taskletv2.GenericBinary{Payload: data}
		serialized, err := proto.Marshal(msg)
		if err != nil {
			return err
		}
		ex.Spec.Input.SerializedData = serialized
	}
	fmt.Println(protojson.Format(ex))

	if err := ytd.RegisterExecution(rootCtx, bb, ex, ""); err != nil {
		return err
	}
	opID, err := ytd.Spawn(rootCtx, consts.ExecutionID(ex.Meta.Id))
	if err != nil {
		return err
	}
	fmt.Println(opID)
	return nil
}

var ytRunCmd = &cobra.Command{
	Use:   "ytrun",
	Short: "TEST TEST TEST",
	RunE:  ytRunHandler,
}

var (
	inputFile string
	buildID   string
)

func init() {
	ytRunCmd.Flags().StringVar(&inputFile, "input", "", "bytes input")
	ytRunCmd.Flags().StringVar(&buildID, "build", "1f42441e-07d0-4ea3-ad12-c9df9c51075a", "buildID")
}
