package server

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/experimental/internal/cli"
	"a.yandex-team.ru/tasklet/experimental/internal/cmd/server/migrations"
	"a.yandex-team.ru/tasklet/experimental/internal/lib"
	"a.yandex-team.ru/tasklet/experimental/internal/locks"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/ydbstore"
	"a.yandex-team.ru/tasklet/experimental/internal/utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
	"a.yandex-team.ru/tasklet/experimental/internal/ydbmigrate"
)

var (
	purgeFlag bool
)

var initMessageTemplate = `Initializing database
	Endpoint: %q
	Database: %q
	Folder: %q
	PurgeOld: %v
`

var migrateMessageTemplate = `Migrating database
	Endpoint: %q
	Database: %q
	Folder: %q
	BaseSchemaVersion: %v
	TargetSchemaVersion: %v
`

func storageInitHandler(_ *cobra.Command, _ []string) error {
	loggers, err := lib.NewLoggers(conf.Logging)
	if err != nil {
		return err
	}
	coreLog := loggers[lib.CoreLogger]
	coreLog.Info("Initializing")

	rootCtx, rootCancel := context.WithCancel(context.Background())
	defer rootCancel()

	fmt.Printf(initMessageTemplate, conf.DB.Ydb.Endpoint, conf.DB.Ydb.Database, conf.DB.Ydb.Folder, purgeFlag)
	if !forceYesFlag {
		if !cli.AskForConfirmation("confirm action", coreLog) {
			return nil
		}
	}

	c, err := xydb.NewClient(
		rootCtx,
		conf.DB.Ydb,
		utils.MustToken(utils.LoadToken(conf.DB.Ydb.TokenPath)),
		coreLog,
		nil,
	)
	if err != nil {
		coreLog.Errorf("Failed to construct YDB. Err: %v", err)
		return err
	}
	c.SetLogQueries(true)
	if purgeFlag {
		if strings.Contains(c.Folder, "tasklets") || strings.TrimSpace(c.Folder) == "" {
			return xerrors.Errorf("Failsafe: bad folder name for purge: %q", c.Folder)
		}
		coreLog.Warnf("Purging database. Storage: %q", c.GetPrefix())
		if err := ydbstore.PurgeDatabase(rootCtx, c); err != nil {
			coreLog.Errorf("Failed to purge old tables. Err: %v", err)
			return err
		}
		if err := ydbmigrate.PurgeMetaTables(rootCtx, c); err != nil {
			return xerrors.Errorf("Failed to purge meta tables: %w", err)
		}

	}
	locksRepo := locks.NewLocksRepo(c)
	if err := locksRepo.ResetLocksTable(); err != nil {
		return err
	}
	if err := locksRepo.ResetExcludesTable(); err != nil {
		return err
	}

	if err := ydbmigrate.CreateMetaTables(rootCtx, c); err != nil {
		return xerrors.Errorf("Failed create meta tables: %w", err)
	}

	if err := ydbstore.CreateTables(rootCtx, c); err != nil {
		return xerrors.Errorf("Failed create tables: %w", err)
	}

	mc := ydbmigrate.NewMigrationClient(c, coreLog)

	for v := ydbmigrate.NilSchemaVersion; v < ydbstore.SchemaVersion; v = v + 1 {
		err := mc.CompleteMigration(
			rootCtx, &ydbmigrate.Migration{
				BaseVersion: v,
				Description: fmt.Sprintf("initialization v%d->v%d", v, v+1),
			},
		)
		if err != nil {
			return xerrors.Errorf("initialization failed for version %d: %w", v, err)
		}
	}

	return nil
}

func storageMigrateHandler(_ *cobra.Command, _ []string) error {
	loggers, err := lib.NewLoggers(conf.Logging)
	if err != nil {
		return err
	}
	coreLog := loggers[lib.CoreLogger]
	coreLog.Info("Initializing migrations")

	rootCtx, rootCancel := context.WithCancel(context.Background())
	defer rootCancel()

	c, err := xydb.NewClient(
		rootCtx,
		conf.DB.Ydb,
		utils.MustToken(utils.LoadToken(conf.DB.Ydb.TokenPath)),
		coreLog,
		nil,
	)
	if err != nil {
		coreLog.Errorf("Failed to construct YDB. Err: %v", err)
		return err
	}
	c.SetLogQueries(true)

	mc := ydbmigrate.NewMigrationClient(c, coreLog)

	currentVersionInfo, err := mc.GetSchemaVersion(rootCtx)
	if err != nil {
		return xerrors.Errorf("Failed to get current db version: %w", err)
	}
	if currentVersionInfo.Version == ydbstore.SchemaVersion {
		coreLog.Info("nothing to do")
		return nil
	}

	fmt.Printf(
		migrateMessageTemplate,
		conf.DB.Ydb.Endpoint,
		conf.DB.Ydb.Database,
		conf.DB.Ydb.Folder,
		currentVersionInfo.Version,
		ydbstore.SchemaVersion,
	)
	if !forceYesFlag {
		if !cli.AskForConfirmation("confirm action", coreLog) {
			return nil
		}
	}

	for currentVersionInfo.Version != ydbstore.SchemaVersion {
		migration := migrations.Migrations[currentVersionInfo.Version]
		if migration == nil {
			coreLog.Errorf("nil migration for v%d", currentVersionInfo.Version)
			return xerrors.Errorf("nil migration for v%d", currentVersionInfo.Version)
		}

		myLease := ydbmigrate.NewMigrationLease()
		for idx := range migration.Mutations {
			coreLog.Infof(
				"Acquiring mutation lease. Idx: %v, Description: %v",
				idx,
				migration.Mutations[idx].MutationName,
			)
			// NB: Acquire lease loop
			for {
				lease, completed, err := mc.GetMutationLease(rootCtx, myLease, migration, idx)
				if err != nil {
					return xerrors.Errorf("Failed to get mutation lease: %w", err)
				}
				if completed {
					coreLog.Infof("Mutation %d already applied. Skipping", idx)
				}
				if lease == myLease {
					coreLog.Info("Lease acquired")
					break
				}
				coreLog.Infof("Concurrent lease. My: %q, Concurrent: %q", myLease.String(), lease.String())
				<-time.After(time.Second * 5)
			}

			coreLog.Infof(
				"Applying mutation. Idx: %v, Description: %v",
				idx,
				migration.Mutations[idx].MutationName,
			)

			if err := migration.Mutations[idx].Func(rootCtx, c); err != nil {
				coreLog.Errorf("Failed to apply mutation: %+v", err)
				return err
			}

			if err := mc.CompleteMutation(rootCtx, myLease, migration, idx); err != nil {
				coreLog.Errorf("Failed to complete mutation: %+v", err)
				return err
			}
			coreLog.Infof(
				"Mutation applied. Idx: %v, Description: %v",
				idx,
				migration.Mutations[idx].MutationName,
			)
		}
		coreLog.Infof(
			"Completing migration. CurrentVersion: %v, Description: %q ",
			migration.GetTargetVersion(),
			migration.Description,
		)

		if err := mc.CompleteMigration(rootCtx, migration); err != nil {
			coreLog.Errorf("Failed to complete migration: %+v", err)
			return err
		}
		currentVersionInfo, err = mc.GetSchemaVersion(rootCtx)
		if err != nil {
			return xerrors.Errorf("Failed to get current db version: %w", err)
		}
	}
	return nil

}

var storageCmd = &cobra.Command{
	Use:   "storage",
	Short: "Init or migrate storage",
	RunE:  nil,
}

var storageInitCmd = &cobra.Command{
	Use:     "init",
	Short:   "Initialize database",
	PreRunE: validateConfig,
	RunE:    storageInitHandler,
}

var storageMigrateCmd = &cobra.Command{
	Use:     "migrate",
	Short:   "Migrate database",
	PreRunE: validateConfig,
	RunE:    storageMigrateHandler,
}

var forceYesFlag bool

func init() {
	storageCmd.PersistentFlags().BoolVar(&forceYesFlag, "force-yes", false, "do not ask for confirmation")
	storageCmd.AddCommand(storageInitCmd)
	storageCmd.AddCommand(storageMigrateCmd)
	storageInitCmd.Flags().BoolVar(&purgeFlag, "purge", false, "purge old tables")
}
