package wrench

import (
	"fmt"

	"github.com/spf13/cobra"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/tasklet/experimental/cmd/wrench/configs"
)

var (
	conf   *Config
	logger log.Logger
)

func rootPreRunHandler(_ *cobra.Command, _ []string) error {
	logger.Debugf("Loading config: %v", targetCluster)
	bytes, err := configs.GetConfig(targetCluster)
	if err != nil {
		return xerrors.Errorf("Failed to load config: %w", err)
	}
	config, err := NewConfigFromBytes(bytes)
	if err != nil {
		return xerrors.Errorf("Failed to parse config: %w", err)
	}
	if err := config.Validate(); err != nil {
		logger.Error("Config validation failed", log.Error(err))
	}
	conf = config
	return nil
}

var rootCmd = &cobra.Command{
	Use:               "tasklet-wrench",
	Short:             "Tasklet wrench tool",
	PersistentPreRunE: rootPreRunHandler,
}

func Execute() {
	maxprocs.AdjustAuto()
	logConf := zap.ConsoleConfig(log.DebugLevel)
	logConf.EncoderConfig.EncodeLevel = zapcore.CapitalColorLevelEncoder
	logger = zap.Must(logConf)

	if err := rootCmd.Execute(); err != nil {
		logger.Error("Error", log.Error(err))
	}
}

const defaultCluster = "test"
const defaultTokenPath = "~/.tasklet/token"

var targetCluster string
var tokenPath string

func init() {
	rootCmd.PersistentFlags().PrintDefaults()
	rootCmd.PersistentFlags().StringVarP(
		&targetCluster,
		"cluster",
		"c",
		defaultCluster,
		fmt.Sprintf("name of base embedded config; values: %v", configs.ListConfigs()),
	)
	rootCmd.PersistentFlags().StringVarP(
		&tokenPath,
		"token",
		"t",
		defaultTokenPath,
		"token path",
	)
}
