package executor

import (
	"context"
	"errors"
	"net"

	"github.com/grpc-ecosystem/go-grpc-middleware"
	"github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	"github.com/grpc-ecosystem/go-grpc-middleware/recovery"
	uberZap "go.uber.org/zap"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials"
	"google.golang.org/grpc/reflection"
	"google.golang.org/protobuf/proto"

	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics/nop"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/sandbox/common/go/clients"
	privatetaskletv1 "a.yandex-team.ru/tasklet/api/priv/v1"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/prototools"
	"a.yandex-team.ru/tasklet/experimental/internal/requestctx"
	"a.yandex-team.ru/tasklet/experimental/internal/xgrpc"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

type App struct {
	l                      log.Logger
	serverClients          *apiclient.ServerConnection
	execution              *executionInfo
	sandboxSupport         *sandboxSupportAPIClient
	userServer             *grpc.Server
	sandboxExternalSession sandbox.SandboxExternalSession
}

func NewApp(logger log.Logger, session sandbox.SandboxExternalSession) *App {
	return &App{l: logger, sandboxExternalSession: session}
}

func (a *App) initServerConnection(conf *apiclient.Config, executionID consts.ExecutionID) (err error) {
	var auth credentials.PerRPCCredentials
	if conf.EnableAuth {
		if a.sandboxExternalSession == "" {
			return xerrors.New("Missing credentials")
		}
		auth = apiclient.NewCredentials(a.sandboxExternalSession.String(), consts.ExternalSandboxSessionMethod)
	} else {
		auth = apiclient.NewInsecureTestCredentials(requestctx.ExecutionSubject, "", executionID)
	}
	a.serverClients, err = apiclient.NewServerConnection(
		a.l.WithName("serverClients"),
		conf,
		auth,
	)
	return
}

func (a *App) initExecutionInfo(ctx context.Context, executionID string) error {
	info := &executionInfo{}
	client := a.serverClients.InternalService
	a.l.Infof("Requesting execution info. ExecutionID: %q", executionID)
	if getExecutionResp, err := client.BootstrapExecution(
		ctx,
		&privatetaskletv1.BootstrapExecutionRequest{Id: executionID},
	); err != nil {
		return err
	} else {
		info.execution = getExecutionResp.GetExecution()
		info.build = getExecutionResp.GetBuild()
	}
	a.execution = info

	return nil
}

func (a *App) initSandboxSupport() error {
	if !isSandbox() {
		a.l.Info("Not enabling sandbox support API: not a Sandbox")
		return nil
	}
	api, err := newSandboxSupportAPIClient(a.l.WithName("sb_support"))
	if err != nil {
		return err
	}
	a.sandboxSupport = api
	return nil
}

func (a *App) initIOMessages(ctx context.Context) error {
	client := a.serverClients.SchemaRegistry
	schema := a.execution.build.GetSpec().GetSchema()

	a.l.Infof(
		"Preparing IO Messages. SchemaID: %q, InputMsg: %q, OutputMsg: %q",
		schema.GetSimpleProto().GetSchemaHash(),
		schema.GetSimpleProto().GetInputMessage(),
		schema.GetSimpleProto().GetOutputMessage(),
	)

	simpleSchema := schema.GetSimpleProto()
	req := &taskletv2.GetSchemaRequest{Hash: simpleSchema.GetSchemaHash()}

	response, err := client.GetSchema(ctx, req)
	if err != nil {
		return xerrors.Errorf("failed to get schema with ID %q: %w", req.Hash, err)
	}

	inputMessage, outputMessage, err := prototools.MakeIOMessages(response.GetSchema(), simpleSchema)
	if err != nil {
		return xerrors.Errorf("failed to initialize protobuf resolver: %w", err)
	}

	{
		// NB: Just check unmarshalling
		err := proto.Unmarshal(a.execution.getSerializedInput(), inputMessage)
		if err != nil {
			return xerrors.Errorf("Failed to unmarshall input: %w", err)
		}
	}
	a.execution.inputMessage = inputMessage
	a.execution.outputMessage = outputMessage
	return nil
}

func (a *App) initUserServer() error {
	serverLogger := a.l.WithName("grpc_server")

	var unaryInterceptors []grpc.UnaryServerInterceptor
	unaryInterceptors = append(
		unaryInterceptors,
		grpc_recovery.UnaryServerInterceptor(
			grpc_recovery.WithRecoveryHandlerContext(
				xgrpc.PanicHandler(serverLogger, false),
			),
		),
	)

	if grpcServerLog, ok := serverLogger.(*zap.Logger); ok {
		zapOpts := []grpc_zap.Option{
			grpc_zap.WithLevels(grpc_zap.DefaultCodeToLevel),
		}
		unaryInterceptors = append(
			unaryInterceptors,
			grpc_zap.UnaryServerInterceptor(grpcServerLog.L.WithOptions(uberZap.AddCallerSkip(1)), zapOpts...),
		)
	}

	serverOptions := make([]grpc.ServerOption, 0)

	// Unary interceptor
	serverOptions = append(
		serverOptions,
		grpc_middleware.WithUnaryServerChain(unaryInterceptors...),
	)

	a.userServer = grpc.NewServer(serverOptions...)
	reflection.Register(a.userServer)
	return nil
}

func (a *App) buildLocalServerHandler(ctxProvider *ContextProvider) error {
	sandboxClient, err := sandbox.New(
		&sandbox.Config{
			Host: clients.DefaultHost,
		},
		a.sandboxExternalSession.String(),
		a.l.WithName("sandbox"),
		&nop.Registry{},
	)
	if err != nil {
		return err
	}

	localServer := NewLocalServiceHandler(ctxProvider, sandboxClient, a.l.WithName("local_server"))
	RegisterLocalServiceHandler(a.userServer, localServer)
	return nil
}

func (a *App) serveGrpc(ctx context.Context, ipv6Listener, ipv4Listener net.Listener) error {

	servingSuccess := errors.New("success")

	serveForListenerFunc := func(listener net.Listener) error {
		err := a.userServer.Serve(listener)
		if err != nil {
			a.l.Errorf("gRPC server on %v returned error: %+v", listener.Addr(), err)
		} else {
			err = servingSuccess
			a.l.Infof("gRPC server on %v stopped", listener.Addr())
		}
		return err
	}

	servingGroup, servingCtx := errgroup.WithContext(ctx)

	servingGroup.Go(func() error { return serveForListenerFunc(ipv6Listener) })
	servingGroup.Go(func() error { return serveForListenerFunc(ipv4Listener) })
	servingGroup.Go(func() error {
		<-servingCtx.Done()
		a.userServer.Stop()
		return servingCtx.Err()
	})

	err := servingGroup.Wait()
	if err == servingSuccess {
		err = nil
	}
	return err
}

func (a *App) buildListenSockets() (net.Listener, net.Listener, error) {
	a.l.Info("Looking for port fot executor service")
	try := 0
	for ; try < 30; try++ {
		listener6, err := net.Listen("tcp", "[::1]:0") // choose port automatically
		if err != nil {
			continue
		}
		_, port, err := net.SplitHostPort(listener6.Addr().String())
		if err != nil {
			panic(err)
		}
		listener4, err := net.Listen("tcp", "127.0.0.1:"+port)
		if err != nil {
			_ = listener6.Close()
			continue
		}
		a.l.Infof("Listening on port %v (after %v tries): %v, %v", port, try+1, listener6.Addr(), listener4.Addr())
		return listener6, listener4, nil
	}
	return nil, nil, xerrors.Errorf("Failed to find free port after %v tries", try)
}

// Initialize runs common initialization sequence
func (a *App) Initialize(ctx context.Context, args Args) error {

	if err := a.initServerConnection(args.Config, consts.ExecutionID(args.ExecutionID)); err != nil {
		return xerrors.Errorf("Failed to init serverClients connection: %w", err)
	}
	if err := a.initExecutionInfo(ctx, args.ExecutionID); err != nil {
		return xerrors.Errorf("Failed to init execution info: %w", err)
	}
	if err := a.initIOMessages(ctx); err != nil {
		return xerrors.Errorf("Failed to init IO messages: %w", err)
	}
	if err := a.initSandboxSupport(); err != nil {
		return xerrors.Errorf("Failed to init sandbox support API: %w", err)
	}
	if err := a.initUserServer(); err != nil {
		return xerrors.Errorf("Failed to init user GRPC server: %w", err)
	}
	return nil
}
