package executor

import (
	"sync"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/tasklet/api/v2"
)

type ContextProvider struct {
	meta        *taskletv2.ExecutionMeta
	executor    *taskletv2.ExecutorRef
	environment *taskletv2.EnvironmentRef
	schema      *taskletv2.IOSchema
	lock        sync.RWMutex
	produced    bool
}

func NewContextProvider(meta *taskletv2.ExecutionMeta) *ContextProvider {
	rv := &ContextProvider{
		meta:        proto.Clone(meta).(*taskletv2.ExecutionMeta),
		executor:    nil,
		environment: nil,
	}
	return rv
}

func (cp *ContextProvider) Done() {
	cp.lock.Lock()
	defer cp.lock.Unlock()
	if cp.produced {
		panic("double done")
	}
	cp.produced = true
}

func (cp *ContextProvider) GetContext() *taskletv2.Context {
	cp.lock.RLock()
	defer cp.lock.RUnlock()
	if !cp.produced {
		panic("not done")
	}
	return &taskletv2.Context{
		Meta:        proto.Clone(cp.meta).(*taskletv2.ExecutionMeta),
		Executor:    proto.Clone(cp.executor).(*taskletv2.ExecutorRef),
		Environment: proto.Clone(cp.environment).(*taskletv2.EnvironmentRef),
		Schema:      proto.Clone(cp.schema).(*taskletv2.IOSchema),
	}
}

func (cp *ContextProvider) SetExecutorRef(executor *taskletv2.ExecutorRef) {
	cp.lock.Lock()
	defer cp.lock.Unlock()
	if cp.produced {
		panic("updating built context")
	}
	if cp.executor != nil {
		panic("double write on executor")
	}
	cp.executor = proto.Clone(executor).(*taskletv2.ExecutorRef)
}

func (cp *ContextProvider) SetEnvironmentRef(env *taskletv2.EnvironmentRef) {
	cp.lock.Lock()
	defer cp.lock.Unlock()
	if cp.produced {
		panic("updating built context")
	}
	if cp.environment != nil {
		panic("double write on executor")
	}
	cp.environment = proto.Clone(env).(*taskletv2.EnvironmentRef)
}

func (cp *ContextProvider) SetIOSchema(env *taskletv2.IOSchema) {
	cp.lock.Lock()
	defer cp.lock.Unlock()
	if cp.produced {
		panic("updating built context")
	}
	if cp.schema != nil {
		panic("double write on executor")
	}
	cp.schema = proto.Clone(env).(*taskletv2.IOSchema)
}
