package executor

import (
	"context"
	"encoding/base64"
	"fmt"
	"os"
	"path/filepath"
	"testing"
	"time"

	"github.com/golang/protobuf/proto"
	"github.com/stretchr/testify/suite"
	"google.golang.org/protobuf/types/dynamicpb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
)

type ExecuteTaskletSuite struct {
	suite.Suite
	objects *testutils.ObjectGenerator
	tasklet *taskletv2.Tasklet
	logger  log.Logger
	tempdir string
}

func (es *ExecuteTaskletSuite) SetupSuite() {
	es.objects = testutils.NewObjectGenerator()
	og := es.objects
	ns := og.NewNamespace("foo_ns")
	es.tasklet = og.NewTasklet("bar_tl", ns)
}

func (es *ExecuteTaskletSuite) SetupTest() {
	es.tempdir = testutils.TwistTmpDir(es.T())
	zapConf := zap.ConsoleConfig(log.DebugLevel)
	if testutils.IsYaTest() {
		zapConf.OutputPaths = []string{filepath.Join(es.tempdir, "run_tasklet.log")}
	} else {
		zapConf.OutputPaths = append(zapConf.OutputPaths, filepath.Join(es.tempdir, "run_tasklet.log"))
	}
	es.logger = zap.Must(zapConf)

}

func (es *ExecuteTaskletSuite) writeBinary(text string, binPath string) {

	file, err := os.OpenFile(binPath, os.O_CREATE|os.O_RDWR|os.O_TRUNC, 0775)
	es.NoError(err)
	defer func() {
		es.NoError(file.Close())
	}()

	n, err := file.WriteString(text)
	es.NoError(err)
	es.Equal(len(text), n)
}

func (es *ExecuteTaskletSuite) setupRun() (*TaskletEnvironment, *executionInfo) {
	build := es.objects.NewBuild(es.tasklet)
	execution := es.objects.NewExecution(build)
	si, err := proto.Marshal(&taskletv2.GenericBinary{Payload: []byte("my_input")})
	es.NoError(err)
	execution.Spec.Input.SerializedData = si

	env := &TaskletEnvironment{
		l:           es.logger.WithName("environment"),
		TaskletPath: filepath.Join(es.tempdir, "tasklet.sh"),
		EnvVars:     make(map[string]string, 0),
		JavaBinPath: "",
		Ref:         nil,
		P:           NewPaths(es.tempdir),
		ctxProvider: NewContextProvider(execution.Meta),
	}
	env.ctxProvider.SetExecutorRef(&taskletv2.ExecutorRef{Address: "localhost:1543"})
	env.ctxProvider.Done()
	es.NoError(env.P.Build())
	info := &executionInfo{
		execution:     execution,
		build:         build,
		inputMessage:  dynamicpb.NewMessage((*taskletv2.GenericBinary)(nil).ProtoReflect().Descriptor()),
		outputMessage: dynamicpb.NewMessage((*taskletv2.GenericBinary)(nil).ProtoReflect().Descriptor()),
	}
	return env, info
}

func (es *ExecuteTaskletSuite) TestOK() {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	env, info := es.setupRun()

	so, err := proto.Marshal(&taskletv2.GenericBinary{Payload: []byte("my_output")})
	es.NoError(err)
	binText := fmt.Sprintf(
		`#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
echo "%s"  | /usr/bin/base64 -d > %s
exit 0;
`,
		base64.StdEncoding.EncodeToString(so),
		outputFileName,
	)
	es.writeBinary(binText, env.TaskletPath)

	outcome, err := runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
	es.NoError(err)
	es.Nil(err)
	kind := outcome.result.Kind
	output, ok := kind.(*taskletv2.ProcessingResult_Output)
	es.True(ok)

	es.Equal(int32(0), outcome.stats.ExitCode)
	es.Equal(so, output.Output.SerializedOutput)
}

func (es *ExecuteTaskletSuite) TestCrash() {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	env, info := es.setupRun()

	binText := `#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
exit 8
`
	es.writeBinary(binText, env.TaskletPath)

	outcome, err := runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
	es.NoError(err)
	es.Equal(int32(8), outcome.stats.ExitCode)

	kind := outcome.result.Kind
	serverError, ok := kind.(*taskletv2.ProcessingResult_ServerError)
	es.True(ok)
	es.Contains(serverError.ServerError.Description, "User job error: exit status 8")
	es.Equal(serverError.ServerError.Code, taskletv2.ErrorCodes_ERROR_CODE_CRASHED)
}

func (es *ExecuteTaskletSuite) TestTimout() {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	env, info := es.setupRun()

	binText := `#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
sleep 10
`
	es.writeBinary(binText, env.TaskletPath)

	// NB: monkeypatch timeout
	ttlBackup := subprocessTTL
	subprocessTTL = time.Millisecond
	defer func() {
		subprocessTTL = ttlBackup
	}()

	outcome, err := runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
	es.NoError(err)
	es.Equal(int32(-1), outcome.stats.ExitCode)

	kind := outcome.result.Kind
	serverError, ok := kind.(*taskletv2.ProcessingResult_ServerError)
	es.True(ok)
	es.Contains(serverError.ServerError.Description, "as killed by timeout")
	es.Equal(serverError.ServerError.Code, taskletv2.ErrorCodes_ERROR_CODE_TIMEOUT)

}

func (es *ExecuteTaskletSuite) TestNoOutput() {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	env, info := es.setupRun()

	binText := `#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
exit 0
`
	es.writeBinary(binText, env.TaskletPath)

	outcome, err := runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
	es.NoError(err)
	es.Equal(int32(0), outcome.stats.ExitCode)
	kind := outcome.result.Kind
	serverError, ok := kind.(*taskletv2.ProcessingResult_ServerError)
	es.True(ok)
	es.Contains(serverError.ServerError.Description, "no such file or directory")
	es.Equal(serverError.ServerError.Code, taskletv2.ErrorCodes_ERROR_CODE_BAD_OUTPUT)
}

func (es *ExecuteTaskletSuite) TestBadOutput() {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	env, info := es.setupRun()

	binText := fmt.Sprintf(
		`#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
echo "ABUBABA!!" > %s
exit 0
`, outputFileName,
	)
	es.writeBinary(binText, env.TaskletPath)

	outcome, err := runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
	es.NoError(err)
	kind := outcome.result.Kind
	serverError, ok := kind.(*taskletv2.ProcessingResult_ServerError)
	es.True(ok)
	es.Contains(serverError.ServerError.Description, "cannot parse")
	es.Equal(serverError.ServerError.Code, taskletv2.ErrorCodes_ERROR_CODE_BAD_OUTPUT)

}

func (es *ExecuteTaskletSuite) TestUserError() {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	env, info := es.setupRun()

	so, err := proto.Marshal(&taskletv2.UserError{Description: "OBLOM ERROR", IsTransient: true})
	es.NoError(err)
	binText := fmt.Sprintf(
		`#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
echo "%s"  | /usr/bin/base64 -d > %s
exit 0;
`,
		base64.StdEncoding.EncodeToString(so),
		errorFileName,
	)
	es.writeBinary(binText, env.TaskletPath)

	outcome, err := runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
	es.NoError(err)
	es.Equal(int32(0), outcome.stats.ExitCode)

	kind := outcome.result.Kind
	userError, ok := kind.(*taskletv2.ProcessingResult_UserError)
	es.True(ok)
	es.NotNil(userError.UserError)
	es.Equal("OBLOM ERROR", userError.UserError.Description)
	es.True(userError.UserError.IsTransient)
}

func (es *ExecuteTaskletSuite) TestParentCancel() {
	env, info := es.setupRun()

	binText := `#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
sleep 10
exit 0
`
	es.writeBinary(binText, env.TaskletPath)

	var outcome *ExecutionOutcome
	var err error
	runChan := make(chan struct{})
	ctx, cancel := context.WithCancel(context.Background())
	go func() {
		outcome, err = runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
		close(runChan)
	}()
	<-time.After(30 * time.Millisecond)
	for {
		_, statErr := os.Stat(filepath.Join(env.P.TaskletSandbox, "tasklet.pid"))
		if statErr == nil {
			break
		}
		<-time.After(30 * time.Millisecond)
	}
	cancel()
	<-runChan
	es.Nil(err)
	kind := outcome.result.Kind
	serverError, ok := kind.(*taskletv2.ProcessingResult_ServerError)
	es.True(ok)
	es.Contains(serverError.ServerError.Description, "Aborted")
	es.Equal(serverError.ServerError.Code, taskletv2.ErrorCodes_ERROR_CODE_GENERIC)
	es.Equal(int32(-1), outcome.stats.ExitCode)
}

func (es *ExecuteTaskletSuite) TestParentTimeout() {
	env, info := es.setupRun()

	binText := `#!/bin/bash
echo "$@" > tasklet.argv
echo "$$" > tasklet.pid
sleep 10
exit 0
`
	es.writeBinary(binText, env.TaskletPath)

	var outcome *ExecutionOutcome
	var err error
	runChan := make(chan struct{})
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	go func() {
		outcome, err = runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
		close(runChan)
	}()
	<-runChan
	es.NoError(err)
	kind := outcome.result.Kind
	serverError, ok := kind.(*taskletv2.ProcessingResult_ServerError)
	es.True(ok)
	es.Contains(serverError.ServerError.Description, "Aborted")
	es.Equal(serverError.ServerError.Code, taskletv2.ErrorCodes_ERROR_CODE_GENERIC)
}

func (es *ExecuteTaskletSuite) TestNoBin() {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	env, info := es.setupRun()

	outcome, err := runTaskletSubprocess(ctx, env, info, es.logger.WithName("runner"))
	es.Error(err)
	es.Nil(outcome)
}

func Test_runTaskletSubprocess(t *testing.T) {

	s := &ExecuteTaskletSuite{}
	suite.Run(t, s)
}
