package executor

import (
	"context"
	"os"
	"os/signal"
	"path/filepath"
	"syscall"

	grpcZap "github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	logzap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/valid"
	privatetaskletv1 "a.yandex-team.ru/tasklet/api/priv/v1"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
	"golang.org/x/sync/errgroup"
)

func MakeLogger() *logzap.Logger {
	conf := logzap.ConsoleConfig(log.DebugLevel)
	if _, ok := os.LookupEnv("YT_JOB_ID"); ok {
		conf.OutputPaths = append(conf.OutputPaths, "stderr")
	}
	return logzap.Must(conf)
}

func swapTokens(
	ctx context.Context,
	executionID consts.ExecutionID,
	conf *apiclient.Config,
	logger log.Logger,
) (sandbox.SandboxExternalSession, error) {
	if !conf.EnableAuth {
		return "", nil
	}
	if externalSession, err := sandbox.YTGetExternalSessionFromEnv(logger); err == nil {
		return sandbox.SandboxExternalSession(externalSession), nil
	}

	sandboxSessionToken, err := sandbox.GetSandboxSessionFromEnv(logger)
	if err != nil {
		ctxlog.Info(ctx, logger, "Skip token exchange: auth disabled")
		return "", err
	}
	auth := apiclient.NewCredentials(sandboxSessionToken, consts.SandboxSessionMethod)
	serverClients, err := apiclient.NewServerConnection(logger, conf, auth)
	if err != nil {
		return "", err
	}
	// Setting up intermediate api server client

	resp, err := serverClients.InternalService.GetExternalSession(
		ctx,
		&privatetaskletv1.GetExternalSessionRequest{ExecutionId: executionID.String()},
	)

	if err != nil {
		return "", err
	}
	return sandbox.SandboxExternalSession(resp.Session), nil

}

func Main(args Args) error {
	_ = maxprocs.AdjustAuto()
	ctx := context.Background()

	if err := validateArgs(args); err != nil {
		return err
	}

	logger := MakeLogger()
	logger.Infof("Executor started with args %+v", args)
	grpcZap.ReplaceGrpcLoggerV2(logger.L)

	if cwd, err := os.Getwd(); err != nil {
		return xerrors.Errorf("Cat not ge CWD(): %w", err)
	} else {
		args.Cwd = cwd
	}

	if value, err := filepath.Abs(args.TaskletPath); err != nil {
		return xerrors.Errorf("Failed to get tasklet abspath: %w", err)
	} else {
		args.TaskletPath = value
	}
	if _, err := os.Stat(args.TaskletPath); err != nil {
		return xerrors.Errorf("Failed to stat %q: %w", args.TaskletPath, err)
	}

	session, errSwap := swapTokens(ctx, consts.ExecutionID(args.ExecutionID), args.Config, logger)
	if errSwap != nil {
		return errSwap
	}
	err := Execute(ctx, logger, args, session)
	if err != nil {
		logger.Error("Execution failed", log.Error(err))
	}
	return err
}

func validateArgs(args Args) error {
	var validationCtx = valid.NewValidationCtx()
	if err := valid.Struct(validationCtx, args.Config); err != nil {
		return xerrors.Errorf("Config validation failed: %w", err)
	}
	return nil
}

func Execute(ctx context.Context, logger log.Logger, args Args, session sandbox.SandboxExternalSession) error {
	app := NewApp(logger, session)
	if err := app.Initialize(ctx, args); err != nil {
		return err
	}

	ctxProvider := NewContextProvider(app.execution.execution.Meta)
	ctxProvider.SetIOSchema(app.execution.build.GetSpec().GetSchema())

	if err := app.buildLocalServerHandler(ctxProvider); err != nil {
		return xerrors.Errorf("Failed to init local server handler: %w", err)
	}

	taskletEnv := NewTaskletEnvironment(logger.WithName("environment"), ctxProvider)
	if err := taskletEnv.prepareEnvironment(
		args.TaskletPath,
		args.JavaBinary,
		app.execution,
		args.Cwd,
	); err != nil {
		return xerrors.Errorf("Environment prepare failure: %w", err)
	}

	ipv6Listener, ipv4Listener, err := app.buildListenSockets()
	if err != nil {
		return err
	}
	ctxProvider.SetExecutorRef(&taskletv2.ExecutorRef{Address: ipv6Listener.Addr().String()})
	ctxProvider.Done()

	// Run sequence

	signalChan := make(chan os.Signal, 1)
	signal.Notify(signalChan, syscall.SIGINT, syscall.SIGTERM, syscall.SIGHUP)
	defer signal.Stop(signalChan)

	group, gCtx := errgroup.WithContext(ctx)
	group.Go(
		func() error {
			return app.serveGrpc(gCtx, ipv6Listener, ipv4Listener)
		},
	)
	group.Go(
		func() error {
			select {
			case sig := <-signalChan:
				logger.Infof("got signal: %v", sig)
				return xerrors.Errorf("Got signal: %v", sig)
			case <-gCtx.Done():
				return gCtx.Err()
			}
		},
	)
	errJobsDone := xerrors.NewSentinel("done")
	group.Go(
		func() error {
			if err := taskletEnv.dumpContextFile(); err != nil {
				return xerrors.Errorf("Context dump failed: %w", err)
			}

			logger.Info("Executing tasklet")
			err := buildAndRunAndReportTasklet(
				gCtx,
				taskletEnv,
				app.serverClients.InternalService,
				app.execution,
				logger.WithName("execute"),
			)
			logger.Infof("Execute tasklet returned: %+v", err)
			if err == nil {
				err = errJobsDone
			}
			return err
		},
	)
	rv := group.Wait()
	if rv == errJobsDone {
		return nil
	} else {
		return rv
	}
}
