package executor

import (
	"context"
	"os"
	"strings"

	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	internalpb "a.yandex-team.ru/tasklet/experimental/proto"
)

type sandboxSupportAPIClient struct {
	logger log.Logger
	client internalpb.SandboxSupportAPIClient
}

func (s *sandboxSupportAPIClient) GetResource(ctx context.Context, resourceID int64) (string, error) {
	s.logger.Infof("Requesting sandbox resource. ID: %v", resourceID)
	req := &internalpb.GetResourceRequest{ResourceId: resourceID}
	resp, err := s.client.GetResource(ctx, req)
	if err != nil {
		return "", err
	}
	s.logger.Infof("Resource acquired. ID: %v, Path: %q", resourceID, resp.Path)
	return resp.Path, nil
}

func getAPIAddress() string {
	return strings.TrimSpace(os.Getenv("TASKLET_SUPPORT_API_URL"))
}

func isSandbox() bool {
	return getAPIAddress() != ""
}

func newSandboxSupportAPIClient(logger log.Logger) (*sandboxSupportAPIClient, error) {
	var supportAPIClient internalpb.SandboxSupportAPIClient

	if !isSandbox() {
		return nil, xerrors.New("Failed to sense sandbox runtime")
	}

	options := []grpc.DialOption{
		grpc.WithTransportCredentials(insecure.NewCredentials()),
	}
	conn, err := grpc.Dial(getAPIAddress(), options...)
	if err != nil {
		return nil, xerrors.Errorf("dial failed: %w", err)
	}
	supportAPIClient = internalpb.NewSandboxSupportAPIClient(conn)
	return &sandboxSupportAPIClient{logger, supportAPIClient}, nil
}
