package executor

import (
	"context"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

func RegisterLocalServiceHandler(server *grpc.Server, lsh *LocalServiceHandler) {
	taskletv2.RegisterExecutorServiceServer(server, lsh)
}

// SandboxConnector whitelists client methods to use in LocalServiceHandler
type SandboxConnector interface {
	GetSecretData(ctx context.Context, secRequest sandbox.SecretDataRequest) (
		map[string]sandbox.SecretKeyValue,
		error,
	)
}

type LocalServiceHandler struct {
	contextProvider *ContextProvider
	sandboxClient   SandboxConnector
	logger          log.Logger
}

func NewLocalServiceHandler(
	contextProvider *ContextProvider,
	sandboxClient SandboxConnector,
	logger log.Logger,
) *LocalServiceHandler {
	return &LocalServiceHandler{
		contextProvider: contextProvider,
		sandboxClient:   sandboxClient,
		logger:          logger,
	}
}

func (lsh *LocalServiceHandler) GetContext(
	_ context.Context, _ *taskletv2.GetContextRequest,
) (*taskletv2.GetContextResponse, error) {
	return &taskletv2.GetContextResponse{Context: lsh.contextProvider.GetContext()}, nil
}

func (lsh *LocalServiceHandler) GetSecretRef(
	ctx context.Context,
	request *taskletv2.GetSecretRefRequest,
) (*taskletv2.GetSecretRefResponse, error) {
	// FIXME: validate request
	ref := request.GetRef()
	response, err := lsh.sandboxClient.GetSecretData(
		ctx, sandbox.SecretDataRequest{
			ID:      ref.GetId(),
			Version: ref.GetVersion(),
		},
	)
	if err != nil {
		return nil, err
	}

	rv := &taskletv2.GetSecretRefResponse{}
	value, ok := response[ref.GetKey()]
	if !ok {
		return nil, status.Errorf(
			codes.NotFound,
			"Key missing. ID: %q, Version: %q, Key: %q",
			ref.GetId(),
			ref.GetKey(),
			ref.GetVersion(),
		)
	}
	rv.Value = &taskletv2.SecretValue{
		Value:    value.Value,
		Encoding: value.Encoding,
	}
	return rv, nil
}
