package executor

import (
	"archive/tar"
	"compress/gzip"
	"errors"
	"io"
	"io/fs"
	"os"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

// NB: Zip Slip vulnerability check. See: https://snyk.io/research/zip-slip-vulnerability
func checkExtractPath(filePath string, outDir string) error {
	destinationPath := filepath.Join(outDir, filePath) + string(os.PathSeparator)
	if !strings.HasPrefix(destinationPath, filepath.Clean(outDir)+string(os.PathSeparator)) {
		return xerrors.Errorf("invalid file path: %q", filePath)
	}
	return nil
}

func UnpackTgz(archivePath string, outDir string) (err error) {
	if err = os.Mkdir(outDir, 0755); err != nil {
		if !errors.Is(err, os.ErrExist) {
			return
		}
	}
	fr, err := os.Open(archivePath)
	if err != nil {
		return err
	}
	defer func() {
		closeErr := fr.Close()
		if err == nil {
			err = closeErr
		}
	}()
	zr, err := gzip.NewReader(fr)
	if err != nil {
		return err
	}
	defer func() {
		closeErr := zr.Close()
		if err == nil {
			err = closeErr
		}
	}()

	tr := tar.NewReader(zr)
	for {
		header, err := tr.Next()
		if err == io.EOF {
			break
		}
		if err != nil {
			return err
		}
		destinationPath := filepath.Join(outDir, header.Name)
		if checkErr := checkExtractPath(header.Name, outDir); checkErr != nil {
			return checkErr
		}
		mode := header.FileInfo().Mode()
		switch {
		case mode.IsDir():
			if destinationPath == outDir {
				// NB: support for "." path
				continue
			}
			err := os.Mkdir(destinationPath, mode.Perm())
			if err != nil {
				return err
			}
		case mode.IsRegular():
			file, err := os.OpenFile(destinationPath, os.O_CREATE|os.O_WRONLY, mode.Perm())
			if err != nil {
				return err
			}
			if _, err := io.Copy(file, tr); err != nil {
				_ = file.Close()
				return err
			}
			if err := file.Close(); err != nil {
				return err
			}
		case mode.Type()&fs.ModeSymlink != 0:
			err := os.Symlink(header.Linkname, destinationPath)
			if err != nil {
				return err
			}
		default:
			return xerrors.Errorf("failed to untar %q: unsupported file type: %q", header.Name, mode.String())
		}
	}
	return nil
}
