package grpcvalid

import (
	"fmt"
	"strings"

	"google.golang.org/genproto/googleapis/rpc/errdetails"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/valid"
)

type RequestScope int

const (
	ScopeRequest = iota
	ScopeResponse
)

type FieldViolations []*errdetails.BadRequest_FieldViolation

func (f FieldViolations) AddE(ff FieldPath, e error) FieldViolations {
	if e != nil {
		return append(f, ff.MakeViolationE(e))
	} else {
		return f
	}
}

func (f FieldViolations) AddF(ff FieldPath, format string, a ...interface{}) FieldViolations {
	return append(f, ff.MakeViolationF(format, a...))
}

func (f FieldViolations) Extend(extras FieldViolations) FieldViolations {
	return append(f, extras...)
}

type FieldPath string

func (f FieldPath) Join(next string) FieldPath {
	return f + FieldPath("."+next)
}

func (f FieldPath) MakeViolationE(err error) *errdetails.BadRequest_FieldViolation {
	return &errdetails.BadRequest_FieldViolation{
		Field:       string(f),
		Description: err.Error(),
	}
}

func (f FieldPath) MakeViolationF(format string, a ...interface{}) *errdetails.BadRequest_FieldViolation {
	return &errdetails.BadRequest_FieldViolation{
		Field:       string(f),
		Description: fmt.Sprintf(format, a...),
	}
}

func MakeValidationErrorStatus(v FieldViolations) *status.Status {
	if len(v) == 0 {
		return nil
	}
	s := status.New(
		codes.InvalidArgument,
		"Validation failed",
	)
	st, err := s.WithDetails(
		&errdetails.BadRequest{
			FieldViolations: v,
		},
	)
	if err != nil {
		panic(err)
	}
	return st
}

func ValidateIdentifier(s string) error {

	if err := valid.StringLen(s, 2, 64); err != nil {
		return err
	}

	for _, c := range s {
		if ('Z' < c || c < 'A') && ('z' < c || c < 'a') && ('9' < c || c < '0') && (c != '-') && (c != '_') {
			return valid.ErrInvalidCharacters
		}
	}
	return nil
}

func ValidateID(s string) error {
	return valid.UUIDv4(s)
}

func ValidateAccountID(accountID string) error {
	parts := strings.SplitN(accountID, ":", 2)
	if parts[0] != "abc" {
		return xerrors.Errorf("account ID must start with 'abc:'")
	}
	return valid.StringLen(accountID, 1, 32)
}
