package handler

import (
	"context"
	"errors"

	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"github.com/gofrs/uuid"
	"google.golang.org/genproto/googleapis/rpc/errdetails"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/handler/xmodels"
	"a.yandex-team.ru/tasklet/experimental/internal/prototools"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
)

func (t *APIHandler) CreateBuild(
	ctx context.Context,
	request *taskletv2.CreateBuildRequest,
) (*taskletv2.CreateBuildResponse, error) {

	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateCreateBuildRequest(request); st != nil {
		return nil, st.Err()
	}
	tl, err := t.findTasklet(ctx, request.GetTasklet(), request.GetNamespace(), false)
	if err != nil {
		return nil, err
	}
	ns, err := t.findNamespace(ctx, tl.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}
	if err = t.permissionsChecker.CheckPermissions(
		ctx, userAuth.Login(), acmodel.BaseTaskletWrite,
		&acmodel.AccessData{Namespace: ns, Tasklet: tl},
	); err != nil {
		return nil, err
	}

	build := &taskletv2.Build{
		Meta: &taskletv2.BuildMeta{
			Id:        uuid.Must(uuid.NewV4()).String(),
			CreatedAt: timestamppb.Now(),
			Tasklet:   request.GetTasklet(),
			Namespace: request.GetNamespace(),
			TaskletId: tl.Meta.Id,
			Revision:  1,
		},
		Spec: &taskletv2.BuildSpec{
			Description:      request.GetDescription(),
			ComputeResources: request.GetComputeResources(),
			Payload:          request.GetPayload(),
			LaunchSpec:       request.GetLaunchSpec(),
			Workspace:        request.GetWorkspace(),
			Environment:      request.GetEnvironment(),
			Schema:           request.GetSchema(),
		},
	}
	if lastBuild, err := t.db.GetLastBuild(ctx, tl.Meta.Id); err != nil {
		if !errors.Is(err, common.ErrObjectNotFound) { // NB: First build
			return nil, err
		}
		build.Meta.Revision = 1
	} else {
		build.Meta.Revision = lastBuild.GetMeta().GetRevision() + 1
	}

	if st := xmodels.ValidateBuild(build); st != nil {
		return nil, st.Err()
	}

	// FIXME: cache schemas & proto resolvers
	{
		ioSchema := build.GetSpec().GetSchema().GetSimpleProto()
		schema, err := t.db.GetSchema(ctx, ioSchema.GetSchemaHash())
		if err != nil {
			if errors.Is(err, common.ErrObjectNotFound) {
				return nil, status.Errorf(
					codes.FailedPrecondition,
					"IO schema not registered. SchemaID: %q",
					ioSchema.SchemaHash,
				)
			}
			return nil, err
		}
		if err := prototools.CheckInputOutputSchema(schema.Fds, ioSchema); err != nil {
			errSchema := prototools.ErrSchema
			if errors.As(err, errSchema) {
				st := status.Newf(
					codes.InvalidArgument,
					"Tasklet schema conflicts with registry: %v",
					err,
				)
				st, _ = st.WithDetails(
					&errdetails.BadRequest_FieldViolation{
						Field:       "schema.simple_proto",
						Description: errSchema.Error(),
					},
				)

				return nil, st.Err()
			}
			return nil, err
		}
	}

	if err := t.db.AddBuild(ctx, build); err != nil {
		return nil, err
	}
	return &taskletv2.CreateBuildResponse{Build: build}, nil
}

func (t *APIHandler) GetBuild(
	ctx context.Context,
	request *taskletv2.GetBuildRequest,
) (*taskletv2.GetBuildResponse, error) {
	// FIXME: check read ACL

	if st := xmodels.ValidateGetBuildRequest(request); st != nil {
		return nil, st.Err()
	}

	build, err := t.findBuild(ctx, request.BuildId, false)
	if err != nil {
		return nil, err
	}

	return &taskletv2.GetBuildResponse{Build: build}, nil

}

func (t *APIHandler) ListBuilds(
	ctx context.Context,
	request *taskletv2.ListBuildsRequest,
) (*taskletv2.ListBuildsResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateListBuildsRequest(request); st != nil {
		return nil, st.Err()
	}

	tl, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, false)
	if err != nil {
		return nil, err
	}
	ns, err := t.findNamespace(ctx, tl.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}

	if err = t.permissionsChecker.CheckPermissions(
		ctx, userAuth.Login(), acmodel.BaseNamespaceRead,
		&acmodel.AccessData{Namespace: ns, Tasklet: tl},
	); err != nil {
		return nil, err
	}

	queryOptions := common.ListBuildsQueryOptions{
		Limit: request.Limit,
		Token: request.Token,
	}
	queryOptions.InferDefaults()

	buildList, err := t.db.ListBuilds(ctx, tl.Meta.Id, queryOptions)
	if err != nil {
		return nil, err
	}

	return &taskletv2.ListBuildsResponse{
		Builds: buildList.Builds,
		Token:  buildList.Token,
	}, nil

}
