package handler

import (
	"context"
	"errors"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/tasklet/api/v2"
	storageCommon "a.yandex-team.ru/tasklet/experimental/internal/storage/common"
)

func (t *APIHandler) findNamespace(
	ctx context.Context, namespace string, precondition bool,
) (*taskletv2.Namespace, error) {

	ns, err := t.db.GetNamespaceByName(ctx, namespace)
	if err != nil {
		if errors.Is(err, storageCommon.ErrObjectNotFound) {
			code := codes.NotFound
			if precondition {
				code = codes.FailedPrecondition
			}
			return nil, status.Errorf(code, "Namespace %q not found", namespace)
		}
		return nil, err
	}
	return ns, nil
}

func (t *APIHandler) findTasklet(
	ctx context.Context, taskletName, namespace string, precondition bool,
) (*taskletv2.Tasklet, error) {

	tasklet, err := t.db.GetTaskletByName(ctx, taskletName, namespace)
	if err != nil {
		if errors.Is(err, storageCommon.ErrObjectNotFound) {
			code := codes.NotFound
			if precondition {
				code = codes.FailedPrecondition
			}
			return nil, status.Errorf(
				code,
				"Tasklet %q not found in namespace %q",
				taskletName,
				namespace,
			)
		}
		return nil, err
	}
	return tasklet, nil
}

func (t *APIHandler) findLabel(
	ctx context.Context, tasklet *taskletv2.Tasklet, labelName string,
) (*taskletv2.Label, error) {

	label, err := t.db.GetLabelByName(ctx, tasklet.Meta.Id, labelName)
	if err != nil {
		if errors.Is(err, storageCommon.ErrObjectNotFound) {
			return nil, status.Errorf(
				codes.NotFound,
				"Label %q not found. Namespace: %q, tasklet: %q",
				labelName,
				tasklet.Meta.Namespace,
				tasklet.Meta.Name,
			)
		}
		return nil, err
	}
	return label, nil
}

func (t *APIHandler) findBuild(ctx context.Context, buildID string, precondition bool) (*taskletv2.Build, error) {
	build, err := t.db.GetBuild(ctx, buildID)
	if err != nil {
		if errors.Is(err, storageCommon.ErrObjectNotFound) {
			code := codes.NotFound
			if precondition {
				code = codes.FailedPrecondition
			}
			return nil, status.Errorf(code, "Build %q not found", buildID)
		}
		return nil, err
	}
	return build, nil
}
