package handler

import (
	"context"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/access"
	"a.yandex-team.ru/tasklet/experimental/internal/requestctx"
	"a.yandex-team.ru/tasklet/experimental/internal/storage"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

var (
	_ taskletv2.TaskletServiceServer = (*APIHandler)(nil)
)

type APIHandler struct {
	Log                log.Logger
	Conf               *Config
	db                 storage.IStorage
	groups             sandbox.IGroupCache
	permissionsChecker *access.PermissionsChecker
	sbx                *sandbox.Client
}

func New(
	l log.Logger,
	c *Config,
	s storage.IStorage,
	g sandbox.IGroupCache,
	sbx *sandbox.Client,
	permissionsChecker *access.PermissionsChecker,
) (*APIHandler, error) {
	return &APIHandler{
		Log:                l,
		Conf:               c,
		db:                 s,
		groups:             g,
		permissionsChecker: permissionsChecker,
		sbx:                sbx,
	}, nil
}

func (t *APIHandler) requireUserAuth(ctx context.Context) (requestctx.UserAuth, error) {
	authSubject, err := requestctx.GetAuthSubject(ctx)
	if err != nil {
		return requestctx.UserAuth{}, status.Errorf(codes.Internal, err.Error())
	}
	userAuth, err := authSubject.RequireUserAuth()
	if err != nil {
		return requestctx.UserAuth{}, status.Errorf(codes.PermissionDenied, err.Error())
	}
	return userAuth, nil
}

func (t *APIHandler) requireExecutorAuth(ctx context.Context) (requestctx.ExecutionAuth, error) {
	authSubject, err := requestctx.GetAuthSubject(ctx)
	if err != nil {
		return requestctx.ExecutionAuth{}, status.Errorf(codes.Internal, err.Error())
	}
	executorAuth, err := authSubject.RequireExecutionAuth()
	if err != nil {
		return requestctx.ExecutionAuth{}, status.Errorf(codes.PermissionDenied, err.Error())
	}
	return executorAuth, nil
}

func (t *APIHandler) requireSandboxTaskAuth(ctx context.Context) (requestctx.SandboxTaskAuth, error) {
	authSubject, err := requestctx.GetAuthSubject(ctx)
	if err != nil {
		return requestctx.SandboxTaskAuth{}, status.Errorf(codes.Internal, err.Error())
	}
	taskAuth, err := authSubject.RequireSandboxAuth()
	if err != nil {
		return requestctx.SandboxTaskAuth{}, status.Errorf(codes.PermissionDenied, err.Error())
	}
	return taskAuth, nil
}
