package handler

import (
	"context"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/priv/v1"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

func (t *APIHandler) ReportExecutionStatus(
	ctx context.Context,
	request *privatetaskletv1.ReportExecutionStatusRequest,
) (*privatetaskletv1.ReportExecutionStatusResponse, error) {

	executorAuth, err := t.requireExecutorAuth(ctx)
	if err != nil {
		return nil, err
	}

	// FIXME: validate request

	if executorAuth.ExecutionID() != consts.ExecutionID(request.Id) {
		ctxlog.Errorf(
			ctx,
			t.Log,
			"Execution id mismatch. Requested: %q, Actual: %q",
			request.Id,
			executorAuth.ExecutionID().String(),
		)
		return nil, status.Errorf(codes.Unauthenticated, "Session execution id is not bound to requested execution")
	}

	stats := request.GetStats()
	if stats == nil {
		// NB: fallback to old style report
		stats = request.GetStatus().GetStats()
	}

	result := request.GetProcessingResult()
	if result == nil {
		// NB: fallback to old style report
		if request.GetStatus().GetResult() != nil {
			result = &taskletv2.ProcessingResult{
				Kind: &taskletv2.ProcessingResult_Output{Output: request.GetStatus().GetResult()},
			}
		} else if request.GetStatus().GetError() != nil {
			result = &taskletv2.ProcessingResult{
				Kind: &taskletv2.ProcessingResult_ServerError{
					ServerError: &taskletv2.ServerError{
						Code:        taskletv2.ErrorCodes_ERROR_CODE_GENERIC,
						Description: request.GetStatus().GetError().GetDescription(),
						IsTransient: false,
					},
				},
			}
		} else {
			return nil, status.Errorf(codes.InvalidArgument, "no result or error provided")
		}
	}

	updateStatusFunc := func(status *taskletv2.ExecutionStatus) error {
		if status.Status == taskletv2.EExecutionStatus_E_EXECUTION_STATUS_FINISHED {
			return xerrors.Errorf("Status of execution already reported")
		}
		if status.ProcessingResult != nil || status.Stats != nil {
			return xerrors.Errorf(
				"already reported. HasProcessingResult: %v, HasStats: %v",
				status.ProcessingResult != nil,
				status.Stats != nil,
			)
		}

		status.ProcessingResult = result
		status.Stats = stats
		return nil
	}

	if _, err := t.db.UpdateExecutionStatus(ctx, consts.ExecutionID(request.GetId()), updateStatusFunc); err != nil {
		return nil, err
	} else {
		return &privatetaskletv1.ReportExecutionStatusResponse{}, nil
	}
}

func (t *APIHandler) GetExternalSession(
	ctx context.Context,
	request *privatetaskletv1.GetExternalSessionRequest,
) (*privatetaskletv1.GetExternalSessionResponse, error) {

	// FIXME: validate request

	taskAuth, err := t.requireSandboxTaskAuth(ctx)
	if err != nil {
		return nil, err
	}

	execution, err := t.db.GetExecution(ctx, request.ExecutionId)

	if err != nil {
		return nil, err
	}

	if execution.GetStatus().GetProcessor().GetSandboxTaskId() != taskAuth.TaskID().ToInt() {
		ctxlog.Errorf(
			ctx,
			t.Log,
			"Task id mismatch. Requested: %d, Actual: %d",
			taskAuth.TaskID().ToInt(),
			execution.GetStatus().GetProcessor().GetSandboxTaskId(),
		)

		return nil, status.Errorf(codes.Unauthenticated, "Task session is not bound to requested execution")
	}

	sessionInfo, err := t.sbx.SearchExternalSession(ctx, consts.ExecutionID(request.ExecutionId))
	if err != nil {
		if xerrors.Is(err, sandbox.ErrSandboxNotFound) {
			return nil, status.Errorf(codes.NotFound, "No session. ExecutionID: %q", request.ExecutionId)
		}
		return nil, err
	}

	return &privatetaskletv1.GetExternalSessionResponse{Session: sessionInfo.Token.String()}, nil
}

func (t *APIHandler) BootstrapExecution(
	ctx context.Context,
	request *privatetaskletv1.BootstrapExecutionRequest,
) (*privatetaskletv1.BootstrapExecutionResponse, error) {

	executorAuth, err := t.requireExecutorAuth(ctx)
	if err != nil {
		return nil, err
	}

	// FIXME: validate request

	if executorAuth.ExecutionID() != consts.ExecutionID(request.Id) {
		ctxlog.Errorf(
			ctx,
			t.Log,
			"Execution id mismatch. Requested: %q, Actual: %q",
			request.Id,
			executorAuth.ExecutionID().String(),
		)
		return nil, status.Errorf(codes.Unauthenticated, "Session execution id is not bound to requested execution")
	}

	rv := &privatetaskletv1.BootstrapExecutionResponse{}

	if execution, err := t.db.GetExecution(ctx, request.Id); err != nil {
		if xerrors.Is(err, common.ErrObjectNotFound) {
			return nil, status.Errorf(codes.NotFound, "Execution does not exist. ID: %q", request.GetId())
		}
		return nil, err
	} else {
		rv.Execution = execution
	}

	build, err := t.findBuild(ctx, rv.Execution.Meta.BuildId, false)
	if err != nil {
		return nil, err
	}
	rv.Build = build
	return rv, nil
}
