package handler

import (
	"context"
	"errors"

	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"github.com/gofrs/uuid"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/handler/xmodels"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
)

func (t *APIHandler) CreateLabel(
	ctx context.Context,
	request *taskletv2.CreateLabelRequest,
) (*taskletv2.CreateLabelResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateCreateLabelRequest(request); st != nil {
		return nil, st.Err()
	}
	ctxlog.Infof(ctx, t.Log, "Name: %q, TL: %q, NS: %q", request.Name, request.Tasklet, request.Namespace)
	tasklet, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, true)
	if err != nil {
		return nil, err
	}
	ns, err := t.findNamespace(ctx, tasklet.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}
	if err := t.permissionsChecker.CheckPermissions(
		ctx,
		userAuth.Login(),
		acmodel.BaseTaskletWrite,
		&acmodel.AccessData{Namespace: ns, Tasklet: tasklet},
	); err != nil {
		return nil, err
	}

	if label, err := t.db.GetLabelByName(ctx, tasklet.Meta.Id, request.Name); err != nil {
		if !errors.Is(err, common.ErrObjectNotFound) {
			return nil, err
		}
	} else {
		return nil, status.Errorf(codes.AlreadyExists, "Label exists. Id: %q", label.Meta.Id)
	}

	if buildID := request.GetBuildId(); buildID != "" {
		if _, err = t.findBuild(ctx, buildID, true); err != nil {
			return nil, err
		}
	}

	l := &taskletv2.Label{
		Meta: &taskletv2.LabelMeta{
			Id:        uuid.Must(uuid.NewV4()).String(),
			Name:      request.Name,
			Tasklet:   request.Tasklet,
			Namespace: request.Namespace,
			CreatedAt: timestamppb.Now(),
			TaskletId: tasklet.Meta.Id,
		},
		Spec: &taskletv2.LabelSpec{
			Revision: 0,
			BuildId:  request.BuildId,
		},
	}
	if st := xmodels.ValidateLabel(l); st != nil {
		return nil, st.Err()
	}

	// NB: Checking for tasklet existence and label commit is not transactional. Seems OK for now.
	err = t.db.AddLabel(ctx, l)
	if err != nil {
		if errors.Is(err, common.ErrObjectExists) {
			return nil, status.Errorf(
				codes.AlreadyExists, "Label already exist. Namespace: %q, Tasklet: %q, Label: %q",
				l.Meta.Namespace,
				l.Meta.Tasklet,
				l.Meta.Name,
			)
		}
		return nil, err
	}
	resp := taskletv2.CreateLabelResponse{
		Label: l,
	}
	return &resp, nil
}

func (t *APIHandler) GetLabel(
	ctx context.Context,
	request *taskletv2.GetLabelRequest,
) (*taskletv2.GetLabelResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateGetLabelRequest(request); st != nil {
		return nil, st.Err()
	}

	tasklet, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, true)
	if err != nil {
		return nil, err
	}
	ns, err := t.findNamespace(ctx, tasklet.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}
	if err = t.permissionsChecker.CheckPermissions(
		ctx,
		userAuth.Login(),
		acmodel.BaseTaskletRead,
		&acmodel.AccessData{Namespace: ns, Tasklet: tasklet},
	); err != nil {
		return nil, err
	}

	ctxlog.Debugf(ctx, t.Log, "Loaded tasklet for label. TaskletId: %q", tasklet.Meta.Id)

	label, err := t.findLabel(ctx, tasklet, request.Label)
	if err != nil {
		return nil, err
	}
	return &taskletv2.GetLabelResponse{
		Label: label,
	}, nil
}

func (t *APIHandler) ListLabels(
	ctx context.Context,
	request *taskletv2.ListLabelsRequest,
) (*taskletv2.ListLabelsResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateListLabelsRequest(request); st != nil {
		return nil, st.Err()
	}

	tasklet, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, false)
	if err != nil {
		return nil, err
	}
	ns, err := t.findNamespace(ctx, tasklet.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}
	if err = t.permissionsChecker.CheckPermissions(
		ctx,
		userAuth.Login(),
		acmodel.BaseTaskletRead,
		&acmodel.AccessData{Namespace: ns, Tasklet: tasklet},
	); err != nil {
		return nil, err
	}

	labels, err := t.db.ListLabels(ctx, tasklet.Meta.Id)
	if err != nil {
		return nil, err
	}
	return &taskletv2.ListLabelsResponse{Labels: labels}, nil
}

func (t *APIHandler) MoveLabel(
	ctx context.Context,
	request *taskletv2.MoveLabelRequest,
) (*taskletv2.MoveLabelResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateMoveLabelRequest(request); st != nil {
		return nil, st.Err()
	}

	tasklet, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, false)
	if err != nil {
		return nil, err
	}
	ns, err := t.findNamespace(ctx, tasklet.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}
	if err = t.permissionsChecker.CheckPermissions(
		ctx,
		userAuth.Login(),
		acmodel.BaseTaskletWrite,
		&acmodel.AccessData{Namespace: ns, Tasklet: tasklet},
	); err != nil {
		return nil, err
	}

	if b, err := t.findBuild(ctx, request.NewBuildId, true); err != nil {
		return nil, err
	} else if b.Meta.Tasklet != request.Tasklet || b.Meta.Namespace != request.Namespace {
		return nil, status.Errorf(codes.FailedPrecondition, "Build %q not found", request.GetNewBuildId())
	}

	label, err := t.findLabel(ctx, tasklet, request.Label)
	if err != nil {
		return nil, err
	}

	if label.Spec == nil {
		label.Spec = &taskletv2.LabelSpec{}
	}
	oldBuild := label.Spec.GetBuildId()
	if oldBuild != request.OldBuildId {
		return nil, status.Errorf(
			codes.Aborted,
			"Bad old build ID. Current: %q, Requested: %q",
			oldBuild,
			request.OldBuildId,
		)
	}

	label.Spec.Revision += 1
	label.Spec.BuildId = request.NewBuildId

	if err := t.db.UpdateLabel(ctx, label); err != nil {
		return nil, err
	}
	return &taskletv2.MoveLabelResponse{Label: label}, nil

}

func (t *APIHandler) UpdateLabel(
	ctx context.Context,
	request *taskletv2.UpdateLabelRequest,
) (*taskletv2.UpdateLabelResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateUpdateLabelRequest(request); st != nil {
		return nil, st.Err()
	}

	tasklet, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, false)
	if err != nil {
		return nil, err
	}
	ns, err := t.findNamespace(ctx, tasklet.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}
	if err = t.permissionsChecker.CheckPermissions(
		ctx,
		userAuth.Login(),
		acmodel.BaseTaskletWrite,
		&acmodel.AccessData{Namespace: ns, Tasklet: tasklet},
	); err != nil {
		return nil, err
	}

	label, err := t.findLabel(ctx, tasklet, request.Label)
	if err != nil {
		return nil, err
	}

	if label.GetSpec().GetRevision() != request.GetExpectedRevision() {
		return nil, status.Errorf(
			codes.Aborted,
			"Bad label revision. Current: %v, Requested: %v",
			label.GetSpec().GetRevision(),
			request.GetExpectedRevision(),
		)
	}
	oldRevision := label.GetSpec().GetRevision()

	if buildID := request.GetBuildId(); buildID != "" {
		if _, err = t.findBuild(ctx, buildID, true); err != nil {
			return nil, err
		}
	}
	if label.Spec == nil {
		label.Spec = &taskletv2.LabelSpec{}
	}
	label.Spec.Revision = oldRevision + 1
	label.Spec.BuildId = request.GetBuildId()

	if err := t.db.UpdateLabel(ctx, label); err != nil {
		return nil, err
	}
	return &taskletv2.UpdateLabelResponse{Label: label}, nil
}
