package handler

import (
	"context"
	"errors"

	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"github.com/gofrs/uuid"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"

	privatetaskletv1 "a.yandex-team.ru/tasklet/api/priv/v1"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/handler/xmodels"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
)

func (t *APIHandler) CreateNamespace(
	ctx context.Context,
	request *taskletv2.CreateNamespaceRequest,
) (*taskletv2.CreateNamespaceResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if err := xmodels.ValidateCreateNamespaceRequest(request); err != nil {
		return nil, err.Err()
	}

	ns := &taskletv2.Namespace{
		Meta: &taskletv2.NamespaceMeta{
			Id:        uuid.Must(uuid.NewV4()).String(),
			Name:      request.GetName(),
			AccountId: request.GetAccountId(),
			CreatedAt: timestamppb.Now(),
		},
		Spec: &taskletv2.NamespaceSpec{},
	}
	permErr := acmodel.AddRole(
		acmodel.NamespaceRead,
		userAuth.Login(),
		taskletv2.PermissionsSubject_E_SOURCE_USER,
		&ns.Meta.Permissions,
	)

	if permErr != nil {
		return nil, permErr
	}

	if err := t.db.AddNamespace(ctx, ns); err != nil {
		if errors.Is(err, common.ErrObjectExists) {
			return nil, status.Errorf(codes.AlreadyExists, "Namespace exists. Namespace: %q", ns.Meta.Name)
		}
		return nil, err
	}
	resp := taskletv2.CreateNamespaceResponse{
		Namespace: ns,
	}
	return &resp, nil
}

func (t *APIHandler) ListNamespaces(
	ctx context.Context,
	request *taskletv2.ListNamespacesRequest,
) (*taskletv2.ListNamespacesResponse, error) {
	// NB: all fields are optional
	namespaces, err := t.db.ListNamespaces(ctx, request.GetOwner())
	if err != nil {
		return nil, err
	}
	return &taskletv2.ListNamespacesResponse{Namespaces: namespaces}, nil
}

func (t *APIHandler) GetNamespace(
	ctx context.Context,
	request *taskletv2.GetNamespaceRequest,
) (*taskletv2.GetNamespaceResponse, error) {
	if st := xmodels.ValidateGetNamespaceRequest(request); st != nil {
		return nil, st.Err()
	}

	ns, err := t.findNamespace(ctx, request.Namespace, false)
	if err != nil {
		return nil, err
	}

	return &taskletv2.GetNamespaceResponse{
		Namespace: ns,
	}, nil
}

// UpdateNamespace is part of internal API
func (t *APIHandler) UpdateNamespace(
	ctx context.Context,
	request *privatetaskletv1.UpdateNamespaceRequest,
) (*privatetaskletv1.UpdateNamespaceResponse, error) {
	if st := xmodels.ValidateUpdateNamespaceRequest(request); st != nil {
		return nil, st.Err()
	}
	var updates []common.NamespaceUpdateOperation

	if len(updates) == 0 {
		return nil, status.Errorf(codes.InvalidArgument, "Will not apply zero updates")
	}

	resultNs, err := t.db.UpdateNamespace(ctx, request.Id, updates...)
	if err != nil {
		if errors.Is(err, common.ErrObjectNotFound) {
			return nil, status.Errorf(codes.NotFound, "Namespace with ID %q not found", request.Id)
		}
		return nil, err
	}

	return &privatetaskletv1.UpdateNamespaceResponse{
		Namespace: resultNs,
	}, nil
}
