package handler

import (
	"context"
	"errors"

	acmodel "a.yandex-team.ru/tasklet/experimental/internal/access/model"
	"github.com/gofrs/uuid"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/handler/xmodels"
	"a.yandex-team.ru/tasklet/experimental/internal/storage/common"
)

func (t *APIHandler) CreateTasklet(
	ctx context.Context,
	request *taskletv2.CreateTaskletRequest,
) (*taskletv2.CreateTaskletResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateCreateTaskletRequest(request); st != nil {
		return nil, st.Err()
	}

	tl := &taskletv2.Tasklet{
		Meta: &taskletv2.TaskletMeta{
			Id:        uuid.Must(uuid.NewV4()).String(),
			Name:      request.GetName(),
			Namespace: request.GetNamespace(),
			AccountId: request.GetAccountId(),
			CreatedAt: timestamppb.Now(),
		},
		Spec: &taskletv2.TaskletSpec{
			Revision:      0,
			Catalog:       request.GetCatalog(),
			TrackingLabel: request.GetTrackingLabel(),
			SourceInfo:    request.GetSourceInfo(),
		},
	}

	permErr := acmodel.AddRole(
		acmodel.TaskletOwner, userAuth.Login(), taskletv2.PermissionsSubject_E_SOURCE_USER, &tl.Meta.Permissions,
	)
	if permErr != nil {
		return nil, permErr
	}

	if st := xmodels.ValidateTasklet(tl); st != nil {
		return nil, st.Err()
	}

	if _, err = t.findNamespace(ctx, tl.Meta.Namespace, true); err != nil {
		return nil, err
	}
	// NB: Checking for namespace existence and tasklet commit is not transactional. Seems OK for now.
	if errAdd := t.db.AddTasklet(ctx, tl); errAdd != nil {
		if errors.Is(errAdd, common.ErrObjectExists) {
			return nil, status.Errorf(
				codes.AlreadyExists,
				"Tasklet already exists. Namespace: %q, Tasklet: %q",
				tl.Meta.Namespace,
				tl.Meta.Name,
			)
		}
		return nil, errAdd
	}
	resp := taskletv2.CreateTaskletResponse{
		Tasklet: tl,
	}
	return &resp, nil
}

func (t *APIHandler) UpdateTasklet(
	ctx context.Context,
	request *taskletv2.UpdateTaskletRequest,
) (*taskletv2.UpdateTaskletResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateUpdateTaskletRequest(request); st != nil {
		return nil, st.Err()
	}

	oldTasklet, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, false)
	if err != nil {
		return nil, err
	}

	ns, err := t.findNamespace(ctx, oldTasklet.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}

	if err = t.permissionsChecker.CheckPermissions(
		ctx, userAuth.Login(), acmodel.BaseTaskletOwner,
		&acmodel.AccessData{Namespace: ns, Tasklet: oldTasklet},
	); err != nil {
		return nil, err
	}

	if request.GetExpectedRevision() != oldTasklet.GetSpec().GetRevision() {
		return nil, status.Errorf(
			codes.Aborted,
			"Bad revision. Requested: %v, Current: %v",
			request.GetExpectedRevision(),
			oldTasklet.GetSpec().GetRevision(),
		)
	}

	oldTasklet.Spec.Revision += 1
	oldTasklet.Spec.Catalog = request.GetCatalog()
	oldTasklet.Spec.TrackingLabel = request.GetTrackingLabel()
	oldTasklet.Spec.SourceInfo = request.GetSourceInfo()

	// FIXME: ensure tracking label exists

	if err := xmodels.ValidateTasklet(oldTasklet); err != nil {
		return nil, err.Err()
	}

	// NB: db query also verifies linear revision history.
	// Optimization to single query requires YDB error parsing
	rv, err := t.db.UpdateTasklet(ctx, oldTasklet)
	if err != nil {
		return nil, err
	}
	return &taskletv2.UpdateTaskletResponse{
		Tasklet: rv,
	}, nil
}

func (t *APIHandler) GetTasklet(
	ctx context.Context,
	request *taskletv2.GetTaskletRequest,
) (*taskletv2.GetTaskletResponse, error) {
	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateGetTaskletRequest(request); st != nil {
		return nil, st.Err()
	}

	tasklet, err := t.findTasklet(ctx, request.Tasklet, request.Namespace, false)
	if err != nil {
		return nil, err
	}

	ns, err := t.findNamespace(ctx, tasklet.Meta.Namespace, false)
	if err != nil {
		return nil, err
	}
	if err := t.permissionsChecker.CheckPermissions(
		ctx,
		userAuth.Login(),
		acmodel.BaseTaskletRead,
		&acmodel.AccessData{Namespace: ns, Tasklet: tasklet},
	); err != nil {
		return nil, err
	}

	resp := &taskletv2.GetTaskletResponse{
		Tasklet: tasklet,
	}
	return resp, nil
}

func (t *APIHandler) ListTasklets(
	ctx context.Context,
	request *taskletv2.ListTaskletsRequest,
) (*taskletv2.ListTaskletsResponse, error) {

	userAuth, err := t.requireUserAuth(ctx)
	if err != nil {
		return nil, err
	}

	if st := xmodels.ValidateListTaskletRequest(request); st != nil {
		return nil, st.Err()
	}

	var ns *taskletv2.Namespace
	if v, err := t.findNamespace(ctx, request.Namespace, false); err != nil {
		return nil, err
	} else {
		ns = v
	}

	tasklets, err := t.db.ListTasklets(ctx, request.GetNamespace())
	if err != nil {
		return nil, err
	}

	responseTasklets := make([]*taskletv2.Tasklet, 0)
	for _, tasklet := range tasklets {
		err := t.permissionsChecker.CheckPermissions(
			ctx, userAuth.Login(), acmodel.BaseTaskletRead,
			&acmodel.AccessData{Namespace: ns, Tasklet: tasklet},
		)
		if err != nil {
			continue
		}
		responseTasklets = append(responseTasklets, tasklet)
	}

	return &taskletv2.ListTaskletsResponse{Tasklets: responseTasklets}, nil
}
