package xmodels

import (
	"github.com/c2h5oh/datasize"
	"golang.org/x/exp/slices"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/reflect/protoreflect"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/grpcvalid"
)

func validateBuildMeta(
	field grpcvalid.FieldPath,
	meta *taskletv2.BuildMeta,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if meta == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	if scope == grpcvalid.ScopeResponse {
		errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(meta.Id))
	}
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(meta.Tasklet))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(meta.Namespace))
	if scope == grpcvalid.ScopeResponse {
		errors = errors.AddE(field.Join("tasklet_id"), grpcvalid.ValidateID(meta.TaskletId))
	}

	return errors
}

func validateComputeResources(
	field grpcvalid.FieldPath,
	cr *taskletv2.ComputeResources,
	_ grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if cr == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	errors = errors.AddE(
		field.Join("vcpu_limit"),
		func(cpuLimit uint64) error {
			if cpuLimit < 1000 {
				return xerrors.Errorf("must be greater than 1000 milli-cores")
			}
			if cpuLimit > 64000 {
				return xerrors.Errorf("must be less than 64000 milli-cores")
			}
			return nil
		}(cr.GetVcpuLimit()),
	)

	errors = errors.AddE(
		field.Join("memory_limit"),
		func(memoryLimit uint64) error {
			if memoryLimit < datasize.MB.Bytes()*100 {
				return xerrors.Errorf("must be greater than 100MB")
			}
			if memoryLimit > datasize.GB.Bytes()*32 {
				return xerrors.Errorf("must not be greater than 32GB")
			}
			return nil
		}(cr.GetMemoryLimit()),
	)

	return errors
}

func validateBuildPayload(
	field grpcvalid.FieldPath,
	spec *taskletv2.BuildSpec_Payload,
	_ grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	if spec.SandboxResourceId == 0 {
		errors = errors.AddF(field.Join("sandbox_resource_id"), "required field")
	}
	return errors
}

func validateJDKOptions(
	field grpcvalid.FieldPath,
	opts *taskletv2.LaunchSpec_JDKOptions,
	_ grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if opts.GetMainClass() == "" {
		errors = errors.AddF(field.Join("main_class"), "main class path is required for java")
	}
	return errors
}

func validateBuildLaunchSpec(
	field grpcvalid.FieldPath,
	spec *taskletv2.LaunchSpec,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {

	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}
	if !slices.Contains(consts.AllLaunchTypes, spec.Type) {
		errors = errors.AddF(
			field.Join("type"), "incorrect launch type %q, available types: %v", spec.Type, consts.AllLaunchTypes,
		)
	}
	if slices.Contains(consts.LaunchJavaTypes, spec.Type) {
		errors = errors.Extend(validateJDKOptions(field.Join("jdk"), spec.GetJdk(), scope))
	}

	return errors
}

func validateStorageSize(ss int64) error {
	size := uint64(ss)
	if size < datasize.MB.Bytes()*100 {
		return xerrors.Errorf("must be greater than 100MB")
	}
	if size > datasize.GB.Bytes()*32 {
		return xerrors.Errorf("must not be greater than 32GB")
	}
	return nil
}

func validateBuildWorkspace(
	field grpcvalid.FieldPath,
	spec *taskletv2.BuildSpec_Workspace,
	_ grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		return errors.AddF(field, "required field")
	}
	if spec.StorageClass == taskletv2.EStorageClass_E_STORAGE_CLASS_INVALID {
		errors = errors.AddF(field.Join("storage_class"), "required field")
	}
	errors = errors.AddE(field.Join("storage_size"), validateStorageSize(spec.StorageSize))
	return errors
}

func validateSchema(
	field grpcvalid.FieldPath,
	spec *taskletv2.IOSchema,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		return errors.AddF(field, "required field")
	}
	errors = errors.Extend(validateSimpleSchema(field.Join("simple_proto"), spec.SimpleProto, scope))
	if spec.Proto != nil {
		errors = errors.AddF(field.Join("proto"), "unsupported")
	}
	return errors
}

func validateSimpleSchema(
	field grpcvalid.FieldPath,
	spec *taskletv2.IOSimpleSchemaProto,
	_ grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		return errors.AddF(field, "required field")
	}
	if spec.SchemaHash == "" {
		errors = errors.AddF(field.Join("schema_hash"), "required field")
	}
	validateProtoName := func(field grpcvalid.FieldPath, name string) {
		casted := protoreflect.FullName(name)
		if !casted.IsValid() {
			errors = errors.AddF(field, "not a valid proto name")
		}
	}

	validateProtoName(field.Join("input_message"), spec.InputMessage)
	validateProtoName(field.Join("output_message"), spec.OutputMessage)
	return errors
}

func validateEnvironment(
	field grpcvalid.FieldPath,
	spec *taskletv2.Environment,
	_ grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	_ = field
	_ = spec
	errors := grpcvalid.FieldViolations{}
	return errors
}

func validateBuildSpec(
	field grpcvalid.FieldPath,
	spec *taskletv2.BuildSpec,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	if scope == grpcvalid.ScopeRequest {
		// NB: do not validate responses
		errors = errors.Extend(
			validateComputeResources(field.Join("compute_resources"), spec.ComputeResources, scope),
		)
	}

	errors = errors.Extend(validateBuildPayload(field.Join("payload"), spec.Payload, scope))
	errors = errors.Extend(validateBuildLaunchSpec(field.Join("launch_spec"), spec.LaunchSpec, scope))
	errors = errors.Extend(validateBuildWorkspace(field.Join("workspace"), spec.Workspace, scope))
	errors = errors.Extend(validateEnvironment(field.Join("environment"), spec.Environment, scope))
	errors = errors.Extend(validateSchema(field.Join("schema"), spec.Schema, scope))

	return errors
}

func validateBuild(
	field grpcvalid.FieldPath,
	build *taskletv2.Build,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}

	if build == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	errors = errors.Extend(validateBuildMeta(field.Join("meta"), build.GetMeta(), scope))
	errors = errors.Extend(validateBuildSpec(field.Join("spec"), build.GetSpec(), scope))
	return errors

}

func ValidateBuild(build *taskletv2.Build) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.Extend(validateBuild(field, build, grpcvalid.ScopeResponse))
	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateCreateBuildRequest(req *taskletv2.CreateBuildRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	if req == nil {
		errors = errors.AddF(field, "required field")
		return grpcvalid.MakeValidationErrorStatus(errors)
	}

	// meta
	scope := grpcvalid.RequestScope(grpcvalid.ScopeRequest)
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.Tasklet))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.Namespace))

	// spec
	errors = errors.Extend(
		validateComputeResources(field.Join("compute_resources"), req.ComputeResources, scope),
	)
	errors = errors.Extend(validateBuildPayload(field.Join("payload"), req.Payload, scope))
	errors = errors.Extend(validateBuildLaunchSpec(field.Join("launch_spec"), req.LaunchSpec, scope))
	errors = errors.Extend(validateBuildWorkspace(field.Join("workspace"), req.Workspace, scope))
	errors = errors.Extend(validateEnvironment(field.Join("environment"), req.Environment, scope))
	errors = errors.Extend(validateSchema(field.Join("schema"), req.Schema, scope))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateGetBuildRequest(req *taskletv2.GetBuildRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	if req == nil {
		errors = errors.AddF(field, "required field")
		return grpcvalid.MakeValidationErrorStatus(errors)
	}
	errors = errors.AddE(field.Join("build_id"), grpcvalid.ValidateID(req.GetBuildId()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateListBuildsRequest(req *taskletv2.ListBuildsRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	if req == nil {
		errors = errors.AddF(field, "required field")
		return grpcvalid.MakeValidationErrorStatus(errors)
	}
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	errors = errors.AddE(
		field.Join("limit"),
		func(lim int64) error {
			if lim == 0 {
				return xerrors.Errorf("must be greater than 0")
			}
			if lim > 100 {
				return xerrors.Errorf("must not be greater than 100")
			}
			return nil
		}(req.GetLimit()),
	)

	return grpcvalid.MakeValidationErrorStatus(errors)
}
