package xmodels

import (
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/valid"

	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/grpcvalid"
)

func validateExecutionMeta(
	field grpcvalid.FieldPath,
	meta *taskletv2.ExecutionMeta,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	_ = scope
	errors := grpcvalid.FieldViolations{}
	if meta == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(meta.Id))
	errors = errors.AddE(field.Join("tasklet_id"), grpcvalid.ValidateID(meta.TaskletId))
	errors = errors.AddE(field.Join("build_id"), grpcvalid.ValidateID(meta.BuildId))

	return errors
}

func validateSandboxGroupName(groupName string) error {
	if err := valid.StringLen(groupName, 1, 20); err != nil {
		return err
	}
	firstLetter := groupName[0]
	if firstLetter < 'A' || firstLetter > 'Z' {
		return valid.ErrInvalidPrefix
	}
	for _, c := range groupName {
		if !(('A' <= c && c <= 'Z') || ('0' <= c && c <= '9') || c == '-' || c == '_') {
			return valid.ErrInvalidCharacters
		}
	}
	return nil
}

func validateExecutionAccountID(accountID string) error {
	if accountID == "" {
		return nil
	}
	if err := validateSandboxGroupName(accountID); err == nil {
		return nil
	}
	return xerrors.NewSentinel("invalid execution account")
}

func validateExecutionRequirements(
	field grpcvalid.FieldPath,
	reqs *taskletv2.ExecutionRequirements,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	_ = scope
	errors := grpcvalid.FieldViolations{}
	errors = errors.AddE(field.Join("account_id"), validateExecutionAccountID(reqs.GetAccountId()))
	return errors
}

func validateExecutionSpec(
	field grpcvalid.FieldPath,
	spec *taskletv2.ExecutionSpec,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}
	errors = errors.AddE(field.Join("author"), valid.StringLen(spec.Author, 1, 20))
	errors = errors.AddE(field.Join("referenced_label"), grpcvalid.ValidateIdentifier(spec.ReferencedLabel))
	errors = errors.Extend(validateExecutionRequirements(field.Join("requirements"), spec.GetRequirements(), scope))

	return errors
}

func validateExecution(
	field grpcvalid.FieldPath,
	execution *taskletv2.Execution,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}

	if execution == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	errors = errors.Extend(validateExecutionMeta(field.Join("meta"), execution.GetMeta(), scope))
	errors = errors.Extend(validateExecutionSpec(field.Join("spec"), execution.GetSpec(), scope))
	return errors
}

func ValidateExecution(execution *taskletv2.Execution) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.Extend(validateExecution(field, execution, grpcvalid.ScopeResponse))
	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateExecuteRequest(req *taskletv2.ExecuteRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	if req.GetLabel() != "" {
		errors = errors.AddE(field.Join("label"), grpcvalid.ValidateIdentifier(req.GetLabel()))
	}
	errors = errors.Extend(
		validateExecutionRequirements(field.Join("requirements"), req.GetRequirements(), grpcvalid.ScopeRequest),
	)

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateGetExecutionRequest(req *taskletv2.GetExecutionRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(req.GetId()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateAbortExecutionRequest(req *taskletv2.AbortExecutionRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(req.GetId()))
	errors = errors.AddE(field.Join("reason"), valid.StringLen(req.GetReason(), 1, 2048))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateListExecutionsByTaskletRequest(req *taskletv2.ListExecutionsByTaskletRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateListExecutionsByBuildRequest(req *taskletv2.ListExecutionsByBuildRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("build_id"), grpcvalid.ValidateID(req.GetBuildId()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}
