package xmodels

import (
	"testing"

	"github.com/stretchr/testify/require"
	"google.golang.org/genproto/googleapis/rpc/errdetails"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/grpcvalid"
)

func compareFieldViolations(t *testing.T, got, want grpcvalid.FieldViolations) {
	if len(want) == 0 {
		require.Len(t, got, 0)
	} else {
		for _, fv := range want {
			has := false
			for _, item := range got {
				if item.Field == fv.Field {
					has = true
					break
				}
			}
			require.True(t, has, "Missing violation %+v. Want: %+v. Got: %+v", fv, want, got)
		}
		for _, fv := range got {
			has := false
			for _, item := range want {
				if item.Field == fv.Field {
					has = true
					break
				}
			}
			require.True(t, has, "Extra violation %v", fv)
		}
	}
}

func TestValidateExecutionRequirements(t *testing.T) {

	validReqs := &taskletv2.ExecutionRequirements{
		AccountId: "SANDBOX_GROUP_NAME",
	}

	patchReqs := func(patch *taskletv2.ExecutionRequirements) *taskletv2.ExecutionRequirements {
		obj := proto.Clone(validReqs).(*taskletv2.ExecutionRequirements)
		proto.Merge(obj, patch)
		return obj
	}

	compareFieldViolations(t, validateExecutionRequirements("", validReqs, grpcvalid.ScopeRequest), nil)

	invalidAccountTests := []struct {
		name      string
		accountID string
	}{
		{
			"invalid_group_name",
			"123",
		},
		{
			"invalid_group_name_2",
			"GroupName",
		},
		{
			"too_long_group_name",
			"VERY-VERY-VERY-VERY-LONG-GROUP-NAME",
		},
	}

	for _, tt := range invalidAccountTests {
		t.Run(
			tt.name, func(t *testing.T) {
				tt := tt
				t.Parallel()
				reqs := patchReqs(&taskletv2.ExecutionRequirements{AccountId: tt.accountID})
				got := validateExecutionRequirements("", reqs, grpcvalid.ScopeRequest)
				want := grpcvalid.FieldViolations{
					&errdetails.BadRequest_FieldViolation{Field: ".account_id"},
				}
				compareFieldViolations(t, got, want)
			},
		)
	}
}
