package xmodels

import (
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/grpcvalid"
)

func validateLabelMeta(
	field grpcvalid.FieldPath,
	meta *taskletv2.LabelMeta,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if meta == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	if scope == grpcvalid.ScopeResponse {
		errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(meta.Id))
	}
	errors = errors.AddE(field.Join("name"), grpcvalid.ValidateIdentifier(meta.Name))
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(meta.Tasklet))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(meta.Namespace))
	if scope == grpcvalid.ScopeResponse {
		errors = errors.AddE(field.Join("tasklet_id"), grpcvalid.ValidateID(meta.TaskletId))
	}

	return errors
}

func validateLabelSpec(
	field grpcvalid.FieldPath,
	spec *taskletv2.LabelSpec,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	_ = scope
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	if spec.BuildId != "" {
		errors = errors.AddE(field.Join("build_id"), grpcvalid.ValidateID(spec.BuildId))
	}

	return errors
}

func validateLabel(
	field grpcvalid.FieldPath,
	label *taskletv2.Label,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}

	if label == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	errors = errors.Extend(validateLabelMeta(field.Join("meta"), label.GetMeta(), scope))
	errors = errors.Extend(validateLabelSpec(field.Join("spec"), label.GetSpec(), scope))
	return errors

}

func ValidateLabel(label *taskletv2.Label) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.Extend(validateLabel(field, label, grpcvalid.ScopeResponse))
	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateCreateLabelRequest(req *taskletv2.CreateLabelRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.Namespace))
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.Tasklet))
	errors = errors.AddE(field.Join("name"), grpcvalid.ValidateIdentifier(req.Name))

	if req.BuildId != "" {
		errors = errors.AddE(field.Join("build_id"), grpcvalid.ValidateID(req.BuildId))
	}

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateGetLabelRequest(req *taskletv2.GetLabelRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	errors = errors.AddE(field.Join("label"), grpcvalid.ValidateIdentifier(req.GetLabel()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateListLabelsRequest(req *taskletv2.ListLabelsRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateMoveLabelRequest(req *taskletv2.MoveLabelRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("label"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	if req.GetOldBuildId() != "" {
		errors = errors.AddE(field.Join("old_build_id"), grpcvalid.ValidateID(req.GetOldBuildId()))
	}
	errors = errors.AddE(field.Join("new_build_id"), grpcvalid.ValidateID(req.GetNewBuildId()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateUpdateLabelRequest(req *taskletv2.UpdateLabelRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}
	errors = errors.AddE(field.Join("label"), grpcvalid.ValidateIdentifier(req.GetLabel()))
	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	errors = errors.AddE(field.Join("build_id"), grpcvalid.ValidateID(req.GetBuildId()))
	if len(errors) > 0 {
		return grpcvalid.MakeValidationErrorStatus(errors)
	}
	return nil
}
