package xmodels

import (
	"google.golang.org/grpc/status"

	privatetaskletv1 "a.yandex-team.ru/tasklet/api/priv/v1"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/grpcvalid"
)

func validateNamespaceMeta(field grpcvalid.FieldPath, meta *taskletv2.NamespaceMeta) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if meta == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}
	errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(meta.Id))
	errors = errors.AddE(field.Join("name"), grpcvalid.ValidateIdentifier(meta.Name))
	errors = errors.AddE(field.Join("account_id"), grpcvalid.ValidateAccountID(meta.AccountId))

	return errors
}

func validateNamespaceSpec(field grpcvalid.FieldPath, spec *taskletv2.NamespaceSpec) grpcvalid.FieldViolations {
	_, _ = field, spec
	return nil
}

func ValidateCreateNamespaceRequest(req *taskletv2.CreateNamespaceRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	if req == nil {
		errors = errors.AddF(field, "required field")
		return grpcvalid.MakeValidationErrorStatus(errors)
	}

	errors = errors.AddE(field.Join("name"), grpcvalid.ValidateIdentifier(req.GetName()))
	errors = errors.AddE(field.Join("account_id"), grpcvalid.ValidateAccountID(req.GetAccountId()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateNamespace(ns *taskletv2.Namespace) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	if ns == nil {
		errors = errors.AddF(field, "required field")
		return grpcvalid.MakeValidationErrorStatus(errors)
	}

	errors = errors.Extend(validateNamespaceMeta(field.Join("meta"), ns.GetMeta()))
	errors = errors.Extend(validateNamespaceSpec(field.Join("spec"), ns.GetSpec()))
	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateGetNamespaceRequest(req *taskletv2.GetNamespaceRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))
	return grpcvalid.MakeValidationErrorStatus(errors)
}

// ValidateUpdateNamespaceRequest is part of private API
func ValidateUpdateNamespaceRequest(req *privatetaskletv1.UpdateNamespaceRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(req.GetId()))
	return grpcvalid.MakeValidationErrorStatus(errors)
}
