package xmodels

import (
	"strings"

	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/valid"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/grpcvalid"
)

func validateTaskletMeta(
	field grpcvalid.FieldPath,
	meta *taskletv2.TaskletMeta,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if meta == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	if scope == grpcvalid.ScopeResponse {
		errors = errors.AddE(field.Join("id"), grpcvalid.ValidateID(meta.Id))
	}
	errors = errors.AddE(field.Join("name"), grpcvalid.ValidateIdentifier(meta.Name))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(meta.Namespace))
	errors = errors.AddE(field.Join("account_id"), grpcvalid.ValidateAccountID(meta.AccountId))
	return errors
}

func validateCatalog(s string) error {
	if !strings.HasPrefix(s, "/") {
		return xerrors.Errorf("does not start with \"/\"")
	}
	if err := valid.StringLen(s, 1, 200); err != nil {
		return err
	}

	items := strings.Split(s, "/")
	if len(items) < 2 {
		return xerrors.Errorf("should have at least one folder")
	}
	for _, subCatalog := range items[1:] {
		if err := valid.StringLen(subCatalog, 1, 100); err != nil {
			return xerrors.Errorf("invalid sub-catalog %q: %v", subCatalog, err)
		}
		for _, c := range subCatalog {
			if ('Z' < c || c < 'A') && ('z' < c || c < 'a') && ('9' < c || c < '0') && (c != '-') && (c != '_') {
				return xerrors.Errorf("invalid character: %q", c)
			}
		}
	}
	return nil
}

func validateTaskletSpec(
	field grpcvalid.FieldPath,
	spec *taskletv2.TaskletSpec,
	_ grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if spec == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	errors = errors.AddE(field.Join("catalog"), validateCatalog(spec.Catalog))

	if spec.GetTrackingLabel() != "" {
		// TODO: make required
		errors = errors.AddE(field.Join("tracking_label"), grpcvalid.ValidateIdentifier(spec.TrackingLabel))
	}
	return errors
}

func validateTasklet(
	field grpcvalid.FieldPath,
	tasklet *taskletv2.Tasklet,
	scope grpcvalid.RequestScope,
) grpcvalid.FieldViolations {
	errors := grpcvalid.FieldViolations{}
	if tasklet == nil {
		errors = errors.AddF(field, "required field")
		return errors
	}

	errors = errors.Extend(validateTaskletMeta(field.Join("meta"), tasklet.GetMeta(), scope))
	errors = errors.Extend(validateTaskletSpec(field.Join("spec"), tasklet.GetSpec(), scope))
	return errors
}

func ValidateTasklet(tasklet *taskletv2.Tasklet) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}
	errors = errors.Extend(validateTasklet(field, tasklet, grpcvalid.ScopeResponse))
	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateCreateTaskletRequest(req *taskletv2.CreateTaskletRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("name"), grpcvalid.ValidateIdentifier(req.Name))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.Namespace))
	errors = errors.AddE(field.Join("account_id"), grpcvalid.ValidateAccountID(req.AccountId))

	errors = errors.AddE(field.Join("catalog"), validateCatalog(req.Catalog))
	if req.GetTrackingLabel() != "" {
		// TODO: make required
		errors = errors.AddE(field.Join("tracking_label"), grpcvalid.ValidateIdentifier(req.TrackingLabel))
	}
	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateUpdateTaskletRequest(req *taskletv2.UpdateTaskletRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))

	errors = errors.AddE(field.Join("catalog"), validateCatalog(req.Catalog))
	if req.GetTrackingLabel() != "" {
		// TODO: make required
		errors = errors.AddE(field.Join("tracking_label"), grpcvalid.ValidateIdentifier(req.TrackingLabel))
	}
	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateGetTaskletRequest(req *taskletv2.GetTaskletRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("tasklet"), grpcvalid.ValidateIdentifier(req.GetTasklet()))
	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}

func ValidateListTaskletRequest(req *taskletv2.ListTaskletsRequest) *status.Status {
	field := grpcvalid.FieldPath("")
	errors := grpcvalid.FieldViolations{}

	errors = errors.AddE(field.Join("namespace"), grpcvalid.ValidateIdentifier(req.GetNamespace()))

	return grpcvalid.MakeValidationErrorStatus(errors)
}
