package xmodels

import (
	"testing"
	"time"

	"github.com/gofrs/uuid"
	"github.com/stretchr/testify/require"
	"google.golang.org/genproto/googleapis/rpc/errdetails"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/tasklet/api/v2"
)

func TestValidateTasklet(t *testing.T) {

	validMeta := &taskletv2.TaskletMeta{
		Id:        uuid.Must(uuid.NewV4()).String(),
		Name:      "bar",
		AccountId: "abc:baz",
		CreatedAt: timestamppb.New(time.Now()),
		Namespace: "namespace",
	}

	patchMeta := func(src *taskletv2.TaskletMeta) *taskletv2.TaskletMeta {
		rv := proto.Clone(validMeta).(*taskletv2.TaskletMeta)
		proto.Merge(rv, src)
		return rv
	}
	validSpec := &taskletv2.TaskletSpec{
		Revision:      1,
		Catalog:       "/foo/bar1/12/v8",
		TrackingLabel: "latest",
		SourceInfo: &taskletv2.SourceInfo{
			SpecPath:      "/path/to/t.yaml",
			RepositoryUrl: "a.yandex-team.ru/arcadia",
		},
	}
	patchSpec := func(src *taskletv2.TaskletSpec) *taskletv2.TaskletSpec {
		rv := proto.Clone(validSpec).(*taskletv2.TaskletSpec)
		proto.Merge(rv, src)
		return rv
	}

	tests := []struct {
		name    string
		tasklet *taskletv2.Tasklet
		want    *errdetails.BadRequest_FieldViolation
	}{
		{
			"ok",
			&taskletv2.Tasklet{
				Meta: validMeta,
				Spec: validSpec,
			},
			nil,
		},
		{
			"fail_name",
			&taskletv2.Tasklet{
				Meta: patchMeta(&taskletv2.TaskletMeta{Name: "BO_BO_INVALID!@@#!#!#_NAME"}),
				Spec: validSpec,
			},
			&errdetails.BadRequest_FieldViolation{Field: ".meta.name"},
		},
		{
			"fail_catalog",
			&taskletv2.Tasklet{
				Meta: validMeta,
				Spec: patchSpec(
					&taskletv2.TaskletSpec{
						Catalog: "//I_AM_INVALID!!$!#!@#!@#//123",
					},
				),
			},
			&errdetails.BadRequest_FieldViolation{Field: ".spec.catalog"},
		},
		{
			"fail_catalog_short",
			&taskletv2.Tasklet{
				Meta: validMeta,
				Spec: patchSpec(
					&taskletv2.TaskletSpec{
						Catalog: "/",
					},
				),
			},
			&errdetails.BadRequest_FieldViolation{Field: ".spec.catalog"},
		},
		{
			"ok_catalog_short",
			&taskletv2.Tasklet{
				Meta: validMeta,
				Spec: patchSpec(
					&taskletv2.TaskletSpec{
						Catalog: "/boo",
					},
				),
			},
			nil,
		},
	}
	for _, tt := range tests {
		t.Run(
			tt.name, func(t *testing.T) {
				got := ValidateTasklet(tt.tasklet)

				if tt.want == nil {
					require.Nil(t, got)
					return
				} else {
					require.NotNil(t, got)
				}
				badReq := got.Details()[0].(*errdetails.BadRequest)

				for _, v := range badReq.FieldViolations {
					if v.Field == tt.want.Field {
						return
					}
				}
				require.Failf(t, "No match", "Want: %+v. Got: %+v", tt.want, badReq)
			},
		)
	}
}
