package lib

import (
	"fmt"
	"os"
	"path/filepath"
	"syscall"

	extzap "go.uber.org/zap"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/log/zap/logrotate"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/valid"
)

var (
	CoreLogger = "core"
	GRPCLogger = "grpc"
	HTTPLogger = "proxy"
)

type Loggers map[string]log.Logger

type LoggerConfig struct {
	Level  string   `yaml:"level"`
	Format string   `yaml:"format"`
	Paths  []string `yaml:"paths"`
}

type LoggingConfig struct {
	LogRoot string                  `yaml:"log_root"`
	Loggers map[string]LoggerConfig `yaml:"loggers"`
}

func (c LoggingConfig) Validate(vctx *valid.ValidationCtx) (bool, error) {
	_ = vctx
	var verrs valid.Errors
	for k := range c.Loggers {
		switch k {
		case CoreLogger, GRPCLogger, HTTPLogger:
			continue
		default:
			verrs = append(
				verrs,
				valid.ErrValidation.Wrap(xerrors.Errorf("Unexpected logger name: %v", k)),
			)
		}
	}
	if len(verrs) != 0 {
		return true, verrs
	}
	return true, nil
}

func newConfig(format string, level log.Level) (extzap.Config, error) {
	switch format {
	case "console":
		return zap.ConsoleConfig(level), nil
	case "cli":
		return zap.CLIConfig(level), nil
	case "json":
		return zap.JSONConfig(level), nil
	case "kv":
		return zap.KVConfig(level), nil
	case "deploy":
		cfg := zap.NewProductionDeployConfig()
		cfg.Level = extzap.NewAtomicLevelAt(zap.ZapifyLevel(level))
		return cfg, nil

	default:
		return extzap.Config{}, fmt.Errorf("unknown config format: [%s]", format)
	}
}

func NewLogger(name, logRoot string, config LoggerConfig) (log.Logger, error) {
	var zapConfig extzap.Config

	if level, err := log.ParseLevel(config.Level); err != nil {
		return nil, err
	} else if zapConfig, err = newConfig(config.Format, level); err != nil {
		return nil, err
	}

	zapConfig.OutputPaths = zapConfig.OutputPaths[:0]
	for _, path := range config.Paths {
		if !(path == "stdout" || path == "stderr" || filepath.IsAbs(path)) {
			path = filepath.Join(logRoot, path)
		}
		zapConfig.OutputPaths = append(zapConfig.OutputPaths, path)
	}

	hostname, _ := os.Hostname()
	fields := []log.Field{
		log.String("revision", buildinfo.Info.ArcadiaSourceRevision),
		log.String("host", hostname),
	}
	l := zap.Must(zapConfig).With(fields...).WithName(name)
	return l, nil
}

func NewLoggers(conf *LoggingConfig) (Loggers, error) {
	loggers := make(Loggers)

	for name, logConfig := range conf.Loggers {
		if logger, err := NewLogger(name, conf.LogRoot, logConfig); err != nil {
			return loggers, err
		} else {
			loggers[name] = logger
		}
	}

	return loggers, nil
}

func init() {
	err := logrotate.RegisterLogrotateSink(syscall.SIGHUP)
	if err != nil {
		panic(err)
	}
}
