package locks

import (
	"context"
	"testing"
	"time"

	"github.com/stretchr/testify/suite"

	testutils "a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

type LocksRepoTestSuite struct {
	suite.Suite
	repo LocksRepo
}

func (suite *LocksRepoTestSuite) SetupSuite() {
	ctx, cancel := context.WithTimeout(context.TODO(), time.Second*5)
	defer cancel()
	suite.repo = NewLocksRepo(testutils.MustGetYdbClient(ctx, nil, suite.T().Name()))
}

func (suite *LocksRepoTestSuite) SetupTest() {
	suite.Require().NoError(suite.repo.ResetExcludesTable())
	suite.Require().NoError(suite.repo.ResetLocksTable())
}

func (suite *LocksRepoTestSuite) TestLockingAndUnlocking() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()
	require := suite.Require()

	ts := time.Now().Add(time.Second * 5).Round(time.Second)
	ls, err := suite.repo.TryLock(ctx, "lock1", "owner1", "", ts)

	require.NoError(err)
	require.Equal("owner1", ls.LockedBy)
	suite.EqualValues(1, ls.SequenceNumber)
	suite.Equal(ts, ls.LockedUntil)

	ls, err = suite.repo.TryLock(ctx, "lock1", "owner1", "", ts.Add(time.Second))
	require.NoError(err)
	if suite.Equal("owner1", ls.LockedBy) {
		suite.EqualValues(1, ls.SequenceNumber)
		suite.Equal(ts.Add(time.Second), ls.LockedUntil)
	}

	ls, err = suite.repo.TryLock(ctx, "lock1", "owner2", "", ts)
	if suite.NoError(err) {
		// other owner can't lock, and we've got old state
		suite.Equal("owner1", ls.LockedBy)
		suite.EqualValues(1, ls.SequenceNumber)
		suite.Equal(ts.Add(time.Second), ls.LockedUntil)
	}

	require.NoError(suite.repo.Unlock(ctx, "lock1", "owner2"), "Dropping a lock held by other is a no-op")

	ls, err = suite.repo.TryLock(ctx, "lock1", "owner1", "", ts)
	require.NoError(err)
	suite.Equal("owner1", ls.LockedBy)

	require.NoError(suite.repo.Unlock(ctx, "lock1", "owner1"))
	ls, err = suite.repo.TryLock(ctx, "lock1", "owner2", "", ts)
	require.NoError(err)
	suite.Equal("owner2", ls.LockedBy)
	suite.EqualValues(2, ls.SequenceNumber)
}

func (suite *LocksRepoTestSuite) TestExcludes() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()

	ts := time.Now()

	suite.Require().NoError(suite.repo.Exclude(ctx, "lock1", "sas", ts.Add(time.Second*15)))

	ls, err := suite.repo.TryLock(ctx, "lock1", "owner1", "sas", ts.Add(time.Second))
	suite.Require().NoError(err)
	suite.Require().Empty(ls.LockedBy)

	ls, err = suite.repo.TryLock(ctx, "lock1", "owner2", "man", ts.Add(time.Second))
	suite.Require().NoError(err)
	suite.Require().Equal("owner2", ls.LockedBy) // another dc could do it
	suite.Require().NoError(suite.repo.Unlock(ctx, "lock1", "owner2"))

	suite.Require().NoError(suite.repo.DropExclude(ctx, "lock1", "sas"))
	ls, err = suite.repo.TryLock(ctx, "lock1", "owner1", "sas", ts.Add(time.Second))
	suite.Require().NoError(err)
	suite.Require().Equal("owner1", ls.LockedBy) // now we can do it
}

func TestLocksRepo(t *testing.T) {
	suite.Run(t, new(LocksRepoTestSuite))
}
