package locks

import (
	"context"
	"fmt"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sd"
)

type LockChecker interface {
	IsLocked() bool
	GetLockState() LockState
}

//go:generate mockery --name=Locker --inpackage --case underscore
type Locker interface {
	LockChecker
	Start()
	Stop()

	CallOnLockAcquire(fn func())
	CallOnLockRelease(fn func())
}

type periodicLocker struct {
	repo    LocksRepo
	id      LockID
	owner   string
	cluster string
	log     log.Logger

	stateLock        sync.Mutex
	state            LockState
	acquireCallbacks []func()
	releaseCallbacks []func()
	wg               sync.WaitGroup
	shutdown         chan struct{}
}

func NewLocker(repo LocksRepo, id LockID, cluster string, logger log.Logger) Locker {
	self, err := sd.GetOwnHostname()
	if err != nil {
		panic("couldn't determine own hostname")
	}
	return &periodicLocker{
		repo:     repo,
		id:       id,
		owner:    self,
		cluster:  cluster,
		log:      logger.WithName(fmt.Sprintf("%s_locker", id)),
		shutdown: make(chan struct{}),
	}
}

func (l *periodicLocker) IsLocked() bool {
	return l.GetLockState().LockedBy == l.owner
}

func (l *periodicLocker) GetLockState() LockState {
	l.stateLock.Lock()
	defer l.stateLock.Unlock()
	if l.state.LockedBy != "" && l.state.LockedUntil.Before(time.Now()) {
		l.log.Warnf("Dropping lock state because it's outdated")
		if l.state.LockedBy == l.owner {
			l.onRelease()
		}
		l.state = LockState{}
	}
	return l.state
}

func (l *periodicLocker) CallOnLockAcquire(fn func()) {
	l.stateLock.Lock()
	defer l.stateLock.Unlock()
	l.acquireCallbacks = append(l.acquireCallbacks, fn)
}

func (l *periodicLocker) CallOnLockRelease(fn func()) {
	l.stateLock.Lock()
	defer l.stateLock.Unlock()
	l.releaseCallbacks = append(l.releaseCallbacks, fn)
}

func (l *periodicLocker) Start() {
	l.wg.Add(1)
	go func() {
		defer l.wg.Done()
		l.runMainLoop()
	}()
}

func (l *periodicLocker) Stop() {
	close(l.shutdown)
	l.wg.Wait()
}

func (l *periodicLocker) runMainLoop() {
	timer := time.NewTimer(time.Nanosecond)
	for {
		select {
		case <-l.shutdown:
			if l.IsLocked() {
				l.stateLock.Lock()
				l.onRelease()
				l.stateLock.Unlock()
				l.unlock()
			}
			return
		case <-timer.C:
			l.refreshLock()
			timer.Reset(time.Second)
		}
	}
}

func (l *periodicLocker) refreshLock() {
	ls, err := l.tryLock(time.Second * 5)
	l.stateLock.Lock()
	defer l.stateLock.Unlock()

	var released, acquired bool
	if err != nil {
		l.log.Warnf("failed to lock because of error: %s", err)
		// metricExceptionsCount.Update(1)
		released = l.state.LockedBy == l.owner && l.state.LockedUntil.Before(time.Now())
		if released {
			l.state = ls
		}
	} else {
		released = l.state.LockedBy == l.owner && l.state.SequenceNumber != ls.SequenceNumber
		acquired = ls.LockedBy == l.owner && l.state.SequenceNumber != ls.SequenceNumber
		l.state = ls
	}
	if l.state.LockedBy == l.owner {
		// metricLockAcquired.Update(1)
		_ = l
	} else {
		// metricLockAcquired.Update(0)
		_ = l
	}
	if released {
		if l.state.LockedBy != "" {
			l.log.Infof("lost the lock, now locked by %s until %s", l.state.LockedBy, l.state.LockedUntil)
		} else {
			l.log.Info("lost the lock")
		}
		l.onRelease()
	}
	if acquired {
		l.log.Infof("acquired lock until %s", ls.LockedUntil)
		l.onAcquire()
	}
}

func (l *periodicLocker) onAcquire() {
	for _, c := range l.acquireCallbacks {
		c()
	}
}

func (l *periodicLocker) onRelease() {
	for _, c := range l.releaseCallbacks {
		c()
	}
}

func (l *periodicLocker) tryLock(timeout time.Duration) (LockState, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()
	return l.repo.TryLock(ctx, l.id, l.owner, l.cluster, time.Now().Add(timeout))
}

func (l *periodicLocker) unlock() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	defer cancel()
	if err := l.repo.Unlock(ctx, l.id, l.owner); err != nil {
		l.log.Errorf("failed to release lock %s: %s", l.id, err.Error())
	}
}

type NoopLocker struct {
}

func (l NoopLocker) IsLocked() bool           { return true }
func (l NoopLocker) GetLockState() LockState  { return LockState{} }
func (l NoopLocker) Start()                   {}
func (l NoopLocker) Stop()                    {}
func (l NoopLocker) CallOnLockAcquire(func()) {}
func (l NoopLocker) CallOnLockRelease(func()) {}
