package locks

import (
	"testing"
	"time"

	"go.uber.org/atomic"

	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sd"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestLocker(t *testing.T) {
	locked := atomic.NewBool(false)
	locker := MockLocksRepo{}
	fqdn, _ := sd.GetOwnHostname()
	locker.On("TryLock", mock.Anything, LockID("random"), fqdn, "", mock.Anything).Return(
		LockState{
			LockedBy:       fqdn,
			SequenceNumber: 1,
			LockedUntil:    time.Now().Add(time.Minute),
		}, nil,
	)
	locker.On("Unlock", mock.Anything, LockID("random"), fqdn).Return(nil)

	l := NewLocker(&locker, "random", "", testutils.MakeLogger("locker.log"))
	var onAcquire, onRelease int
	l.CallOnLockAcquire(
		func() {
			onAcquire++
			locked.Store(true)
		},
	)
	l.CallOnLockRelease(
		func() {
			onRelease++
			locked.Store(false)
		},
	)

	l.Start()
	deadline := time.Now().Add(time.Second)
	for !locked.Load() && time.Now().Before(deadline) {
		time.Sleep(time.Millisecond)
	}
	assert.True(t, locked.Load())
	assert.Equal(t, 1, onAcquire)
	assert.Equal(t, 0, onRelease)

	l.Stop()
	deadline = time.Now().Add(time.Second)
	for locked.Load() && time.Now().Before(deadline) {
		time.Sleep(time.Millisecond)
	}
	assert.False(t, locked.Load())
	assert.Equal(t, 1, onAcquire)
	assert.Equal(t, 1, onRelease)

	locker.AssertNumberOfCalls(t, "TryLock", 1)
	locker.AssertNumberOfCalls(t, "Unlock", 1)
}
