package processor

import (
	"context"
	"fmt"
	"math"

	"github.com/go-openapi/strfmt"
	"google.golang.org/protobuf/encoding/prototext"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/ptr"

	"a.yandex-team.ru/sandbox/common/go/models"
	"a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/state"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

const (
	TaskletExecutorParamName = "tasklet_executor_resource"
	TaskletResourceParamName = "tasklet_to_execute_resource"
	ExecutionIDParamName     = "execution_id"
	EndpointAddressParamName = "endpoint_address"
	EndpointSetParamName     = "endpoint_set"
)

func (p *Processor) TryStopSandboxTask(
	ctx context.Context,
	executionID consts.ExecutionID,
	taskID sandbox.SandboxTaskID,
) error {
	session, errSearch := p.sbx.SearchExternalSession(ctx, executionID)
	if errSearch != nil && xerrors.Is(errSearch, sandbox.ErrSandboxNotFound) {
		return nil
	}
	if session.TaskID != taskID {
		ctxlog.Errorf(ctx, p.logger, "Task ID mismatch. Expected: %v, Actual: %v", taskID, session.TaskID)
		return xerrors.New("Bad session query result")
	}

	// NB: Task is running under user privileges and can not be aborted by robot-tasklets.
	// Try using user external session for this.
	return p.sbx.StopTask(ctx, taskID, session.Token)
}

func (p *Processor) TryCloseExternalSession(
	ctx context.Context,
	executionID consts.ExecutionID,
	taskID sandbox.SandboxTaskID,
) error {
	session, errSearch := p.sbx.SearchExternalSession(ctx, executionID)
	if errSearch != nil && xerrors.Is(errSearch, sandbox.ErrSandboxNotFound) {
		return nil
	}
	// NB: taskID may be 0 for YT or other sandbox unbound runtime
	if session.TaskID != taskID {
		ctxlog.Errorf(ctx, p.logger, "Task ID mismatch. Expected: %v, Actual: %v", taskID, session.TaskID)
		return xerrors.New("Bad session query result")
	}
	return p.sbx.DeleteExternalSession(ctx, session.Token)
}

func (p *Processor) TryCreateExternalSession(
	ctx context.Context,
	executionID consts.ExecutionID,
	owner string,
	taskID sandbox.SandboxTaskID,
) error {
	_, errSearch := p.sbx.SearchExternalSession(ctx, executionID)
	if errSearch != nil && !xerrors.Is(errSearch, sandbox.ErrSandboxNotFound) {
		return xerrors.Errorf("session lookup failed: %w", errSearch)
	}
	_, errCreate := p.sbx.CreateExternalSession(ctx, executionID, owner, taskID)
	if errCreate != nil {
		return xerrors.Errorf("session create failed: %w", errCreate)
	}
	return nil
}

func (p *Processor) TryAcquireExternalSession(
	ctx context.Context,
	executionID consts.ExecutionID,
	owner string,
) (sandbox.SandboxExternalSession, error) {
	sessionInfo, errSearch := p.sbx.SearchExternalSession(ctx, executionID)
	if errSearch != nil && !xerrors.Is(errSearch, sandbox.ErrSandboxNotFound) {
		return "", xerrors.Errorf("session lookup failed: %w", errSearch)
	} else if errSearch == nil {
		return sessionInfo.Token, nil
	}

	session, errCreate := p.sbx.CreateExternalSession(ctx, executionID, owner, 0)
	if errCreate != nil {
		return "", xerrors.Errorf("session create failed: %w", errCreate)
	}
	return session, nil

}

func (p *Processor) TryStartSandboxTask(ctx context.Context, taskID sandbox.SandboxTaskID) error {
	return p.sbx.StartTask(ctx, taskID)
}

func (p *Processor) TryCreateSandboxTask(ctx context.Context, e *taskletv2.Execution, b *taskletv2.Build) (
	sandbox.SandboxTaskID,
	error,
) {
	spec, err := p.buildSandboxSpec(ctx, e, b)
	if err != nil {
		return 0, err
	}
	return p.sbx.TryCreateTask(ctx, consts.ExecutionID(e.Meta.Id), spec)
}

func (p *Processor) buildSandboxSpec(ctx context.Context, e *taskletv2.Execution, b *taskletv2.Build) (
	*models.TaskNew,
	error,
) {
	cr := b.Spec.ComputeResources

	taskletExecutorRes, err := state.SandboxState.GetResource(consts.TaskletExecutorResourceType)
	if err != nil {
		return nil, err
	}
	taskletTaskRes, err := state.SandboxState.GetResource(consts.TaskletTaskResourceType)
	if err != nil {
		return nil, err
	}

	taskParameters := []*models.TaskFieldValidateItem{
		{Name: ptr.String(TaskletExecutorParamName), Value: taskletExecutorRes.ID},
		{Name: ptr.String(TaskletResourceParamName), Value: b.GetSpec().GetPayload().GetSandboxResourceId()},
		{Name: ptr.String(ExecutionIDParamName), Value: e.GetMeta().GetId()},
	}

	var endpointParam *models.TaskFieldValidateItem
	if p.executorConf.EndpointSetName != "" {
		endpointParam = &models.TaskFieldValidateItem{
			Name:  ptr.String(EndpointSetParamName),
			Value: p.executorConf.EndpointSetName,
		}
	} else {
		endpointParam = &models.TaskFieldValidateItem{
			Name:  ptr.String(EndpointAddressParamName),
			Value: p.executorConf.EndpointAddress,
		}
	}
	taskParameters = append(taskParameters, endpointParam)

	author := e.GetSpec().GetAuthor()
	owner := e.GetSpec().GetRequirements().GetAccountId()
	if owner == "" {
		owner = author
	}

	data := &models.TaskNew{
		Type:         "TASKLET",
		Description:  fmt.Sprintf("Tasklet launch\n<b>Meta</b>: \n%s", prototext.Format(e.Meta)),
		Author:       author,
		Owner:        owner,
		Priority:     &sandbox.DefaultPriority,
		CustomFields: taskParameters,
		Requirements: &models.TaskRequirementsUpdate{
			TasksResource: taskletTaskRes.ID,
			Cores:         int64(math.Ceil(float64(cr.VcpuLimit) / 1000.0)),
			RAM:           int64(cr.MemoryLimit),
			ClientTags:    "PORTOD",
			PortoLayers:   p.requiredPortoLayers(ctx, b.GetSpec()),
		},
		Tags: []string{
			fmt.Sprintf("tasklet:%v", b.Meta.Tasklet),
		},
		Hints: []string{
			fmt.Sprintf("tasklet:%v", e.Meta.TaskletId),
			fmt.Sprintf("build:%v", e.Meta.BuildId),
		},
		Uniqueness: &models.Uniqueness{
			ExcludedStatuses: make([]string, 0),
			Key:              e.Meta.Id,
		},
		Notifications:   make([]*models.TaskNotifications, 0),
		SuspendOnStatus: make([]string, 0),
	}

	if err := data.Validate(strfmt.Default); err != nil {
		return nil, err
	}

	return data, nil
}

func (p *Processor) requiredPortoLayers(ctx context.Context, spec *taskletv2.BuildSpec) []int64 {
	var (
		resIDs     []int64
		launchType = spec.GetLaunchSpec().GetType()
		java       = spec.GetEnvironment().GetJava()
	)

	// It would be better do not hardcode base layer, but we have to provide it to use delta layers
	resIDs = append(resIDs, consts.FocalFossaContainerResourceID)

	for jdkName, jdk := range map[string]*taskletv2.JdkReq{
		consts.LaunchTypeJava11: java.GetJdk11(),
		consts.LaunchTypeJava17: java.GetJdk17(),
	} {
		if jdk.GetEnabled() || jdkName == launchType {
			if resID, found := consts.Jdk2ResourceID[jdkName]; found {
				resIDs = append(resIDs, resID)
			} else {
				ctxlog.Errorf(ctx, p.logger, "Failed to find resource for %v", jdkName)
			}
		}
	}

	return resIDs
}
