package processor

import (
	"context"
	"testing"

	"github.com/go-openapi/strfmt"
	"github.com/google/go-cmp/cmp"
	"github.com/stretchr/testify/suite"

	"a.yandex-team.ru/sandbox/common/go/models"
	taskletv2 "a.yandex-team.ru/tasklet/api/v2"
	"a.yandex-team.ru/tasklet/experimental/internal/apiclient"
	"a.yandex-team.ru/tasklet/experimental/internal/consts"
	"a.yandex-team.ru/tasklet/experimental/internal/processor/corpus"
	"a.yandex-team.ru/tasklet/experimental/internal/state"
	testutils "a.yandex-team.ru/tasklet/experimental/internal/test_utils"
	"a.yandex-team.ru/tasklet/experimental/internal/yandex/sandbox"
)

type SandboxSpecBuilderTestSuite struct {
	suite.Suite
	p         *Processor
	objects   *testutils.ObjectGenerator
	tasklet   *taskletv2.Tasklet
	build     *taskletv2.Build
	execution *taskletv2.Execution
}

func (suite *SandboxSpecBuilderTestSuite) SetupTest() {
	gen := testutils.NewObjectGenerator()
	ns := gen.NewNamespace("the_namespace")
	tasklet := gen.NewTasklet("the_tasklet", ns)

	state.SandboxState.SetResource(
		consts.TaskletExecutorResourceType, sandbox.ResourceInfo{ID: testutils.DefaultTaskletExecutorResourceID},
	)
	state.SandboxState.SetResource(
		consts.TaskletTaskResourceType, sandbox.ResourceInfo{ID: testutils.DefaultTaskletTaskResourceID},
	)

	suite.build = gen.NewBuild(tasklet)
	suite.execution = gen.NewExecution(suite.build)
}

func (suite *SandboxSpecBuilderTestSuite) TearDownTest() {
	suite.build = nil
	suite.execution = nil
}

func (suite *SandboxSpecBuilderTestSuite) marshalUnmarshalSpec(data *models.TaskNew) *models.TaskNew {
	serialized, err := data.MarshalBinary()
	suite.NoError(err)
	rv := &models.TaskNew{}
	suite.NoError(rv.UnmarshalBinary(serialized))
	return rv
}

func (suite *SandboxSpecBuilderTestSuite) compareTasks(correct, actual *models.TaskNew) {
	castedCorrect := suite.marshalUnmarshalSpec(correct)
	suite.NoError(castedCorrect.Validate(strfmt.Default))

	castedActual := suite.marshalUnmarshalSpec(actual)
	suite.NoError(castedActual.Validate(strfmt.Default))

	// NB: description output includes prototext'ed execution Meta. prototext format is unstable
	// https://a.yandex-team.ru/arc_vcs/vendor/google.golang.org/protobuf/internal/encoding/text/encode.go?rev=r9019672#L227
	castedCorrect.Description = ""
	castedActual.Description = ""
	suite.True(cmp.Equal(castedCorrect, castedActual), cmp.Diff(castedCorrect, castedActual))
}

func (suite *SandboxSpecBuilderTestSuite) TestDefault() {
	generatedSpec, err := suite.p.buildSandboxSpec(context.Background(), suite.execution, suite.build)
	suite.NoError(err)
	correct := corpus.MustGet("default")
	suite.compareTasks(correct, generatedSpec)
}

func (suite *SandboxSpecBuilderTestSuite) TestArcClient() {
	suite.build.Spec.Environment.ArcClient = &taskletv2.ArcClient{Enabled: true}

	generatedSpec, err := suite.p.buildSandboxSpec(context.Background(), suite.execution, suite.build)
	suite.NoError(err)

	correct := corpus.MustGet("arc_client")

	suite.compareTasks(correct, generatedSpec)
}

func (suite *SandboxSpecBuilderTestSuite) TestJava() {
	suite.build.Spec.Environment.Java = &taskletv2.JavaEnvironment{Jdk17: &taskletv2.JdkReq{Enabled: true}}

	generatedSpec, err := suite.p.buildSandboxSpec(context.Background(), suite.execution, suite.build)
	suite.NoError(err)

	correct := corpus.MustGet("jdk17")

	suite.compareTasks(correct, generatedSpec)
}

func TestSandboxSpecBuilderTestSuite(t *testing.T) {
	p := &Processor{
		conf: &Config{
			Enabled: false,
		},
		executorConf: &apiclient.Config{
			EndpointAddress: "",
			EndpointSetName: "endpoint-set-foo.http",
		},
		logger:  testutils.MakeLogger("/dev/null"),
		storage: nil,
		stop:    nil,
		sbx:     nil,
		ytc:     nil,
		locker:  nil,
	}

	s := &SandboxSpecBuilderTestSuite{p: p}
	suite.Run(t, s)
}
